#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "mapchildwindow.h"
#include "util.h"
#include "styledialog.h"
#include "onlinemapdialog.h"
#include "ui_onlinemapdialog.h"

#include <QFileDialog>
#include <QMessageBox>

#include <fstream>

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    m_ui(new Ui::MainWindow),
    m_app_path(QCoreApplication::applicationDirPath())
    {
    m_ui->setupUi(this);
    m_ui->menuFile->setToolTipsVisible(true);
    m_ui->menuFind->setToolTipsVisible(true);
    m_ui->menuHelp->setToolTipsVisible(true);
    m_ui->menuView->setToolTipsVisible(true);
    m_ui->menuRoute->setToolTipsVisible(true);
    m_ui->menuStyle->setToolTipsVisible(true);

    m_print_separator = m_ui->menuFile->addSeparator();
    m_recent_file_separator = m_ui->menuFile->addSeparator();
    for (int i = 0; i < KMaxRecentFiles; i++)
        {
        m_recent_file_action[i] = new QAction(this);
        m_recent_file_action[i]->setVisible(false);
        connect(m_recent_file_action[i],SIGNAL(triggered()),this,SLOT(OpenRecentFileTriggered()));
        m_ui->menuFile->addAction(m_recent_file_action[i]);
        }

    // Set the app icon. On Windows and Mac that's done in other ways (see the end of the .pro file).
#if !defined(Q_OS_WIN32) && !defined(Q_OS_MAC)
    QIcon icon(":/CT_Arrow-RGB-256x256.png");
    QApplication::setWindowIcon(icon);
#endif

    // Set up the file paths.
    if (m_app_path.length() == 0 || m_app_path[m_app_path.length() - 1] != '/')
        m_app_path += '/';

    // Strip the path down to a string ending just before /src/apps/; if no such pattern is found, use the app path
    m_cartotype_source_path = m_app_path;
    int n = m_cartotype_source_path.lastIndexOf("/src/apps/");
    if (n != -1)
        m_cartotype_source_path.truncate(n + 1);
    if (m_cartotype_source_path.length() == 0 || m_cartotype_source_path[m_cartotype_source_path.length() - 1] != '/')
        m_cartotype_source_path += '/';

    /*
    Find the default style sheet.
    Paths are searched in this order:
    the application directory;
    the subdirectory "style" of the application directory;
    the directory "CartoType/src/style" in the development tree.
    */
    QString style_sheet_name("standard.ctstyle");
    QString style_sheet_path = m_app_path + style_sheet_name;
    if (!QFileInfo::exists(style_sheet_path))
        {
        style_sheet_path = m_app_path + "style/" + style_sheet_name;
        if (!QFileInfo::exists(style_sheet_path))
            style_sheet_path = m_cartotype_source_path + "style/" + style_sheet_name;
        }
    m_default_style_sheet_path.Set(style_sheet_path.utf16(),style_sheet_path.length());

    // Disable menu items not used when there is no map window.
    EnableMenuItems();

    // Update the recent file list and remove nonexistent files.
    UpdateRecentFiles(true);

    // If there's a command line argument not starting with '-', load it as a map.
    bool map_loaded = false;
    int arg_count = QCoreApplication::arguments().size();
    for (int i = 1; i < arg_count; i++)
        {
        QString p = QCoreApplication::arguments().at(i);
        if (p[0] != '-')
            {
            LoadMap({p});
            map_loaded = true;
            }
        }

    // If the main window geometry, and the geometry and states of the child windows, have been saved, restore those settings.
    QSettings settings;
    if (!map_loaded)
        {
        settings.beginGroup("mainWindow");
        QByteArray main_window_geometry = settings.value("geometry").toByteArray();
        settings.endGroup();
        if (main_window_geometry.isEmpty())
            setWindowState(Qt::WindowMaximized);
        else
            restoreGeometry(main_window_geometry);

        // Open the saved map windows.
        int saved_map_window_count = settings.beginReadArray("mapWindows");
        for (int index = 0; index < saved_map_window_count; index++)
            {
            settings.setArrayIndex(index);
            QByteArray geometry = settings.value("geometry").toByteArray();
            QString view_state = settings.value("viewState").toString();

            MapSettings map_settings;
            map_settings.m_metric_units = settings.value("metricUnits",true).toBool();
            map_settings.m_night_mode = settings.value("nightMode",false).toBool();
            map_settings.m_monochrome = settings.value("monochrome",false).toBool();
            map_settings.m_debug_layers = settings.value("debugLayers",false).toBool();
            map_settings.m_fixed_labels = settings.value("fixedLabels",false).toBool();
            map_settings.m_draw_3d_buildings = settings.value("draw3DBuildings",false).toBool();
            map_settings.m_draw_legend = settings.value("drawLegend",false).toBool();
            map_settings.m_draw_scale = settings.value("drawScale",false).toBool();
            map_settings.m_draw_rotator = settings.value("drawRotator",true).toBool();
            map_settings.m_draw_range = settings.value("drawRange",false).toBool();

            std::vector<QString> filename_array;
            int file_count = settings.beginReadArray("files");
            for (int file_index = 0; file_index < file_count; file_index++)
                {
                settings.setArrayIndex(file_index);
                filename_array.push_back(settings.value("name").toString());
                }
            settings.endArray();
            settings.setArrayIndex(index);
            if (file_count == 0)
                filename_array.push_back(settings.value("fileName").toString());
            LoadMap(filename_array,&geometry,&view_state,&map_settings);
            map_loaded = true;
            }
        settings.endArray();
        }

    // Load the first file from the recent file list, or if there aren't any, show a welcome message.
    QStringList files = settings.value("recentFileList").toStringList();
    if (!map_loaded)
        {
        if (files.size() > 0)
            LoadMap({ *files.begin() });
        else
            QMessageBox::about(this,"Welcome to CartoType Maps","The <b>CartoType Maps</b> app allows you to view maps, calculate routes, find places, and more.<br/><br/>"
                               "Please open a map (a .ctm1 file). Some sample maps were provided with this application. "
                               "Right-click on the map to set the start or end of a route, add a pushpin or see what is there.<br/><br/>"
                               "See <a href='http://www.cartotype.com'>cartotype.com</a> for information about creating maps, using the API, and licensing CartoType for your application.");
        }

    // Read the custom route profile from the settings.
    QString profile = settings.value("customRouteProfile").toString();
    if (!profile.isEmpty())
        {
        QByteArray profile_utf8 = profile.toUtf8();
        CartoType::MemoryInputStream input((const uint8_t*)profile_utf8.data(),profile_utf8.length());
        CartoType::RouteProfile p;
        CartoType::Result error = p.ReadFromXml(input);
        if (!error)
            m_custom_route_profile = p;
        }

    // Read the custom style sheet from the settings.
    QString style = settings.value("customStyleSheet").toString();
    if (!style.isEmpty())
        m_custom_style_sheet = style.toStdString();
    }

MainWindow::~MainWindow()
    {
    delete m_ui;
    }

void MainWindow::closeEvent(QCloseEvent* aEvent)
    {
    bool dirty = false;
    bool accept = false;
    QList<QMdiSubWindow*> sub_window_list = m_ui->mdiArea->subWindowList(QMdiArea::StackingOrder);
    for (const auto p : sub_window_list)
        {
        MapChildWindow* w = dynamic_cast<MapChildWindow*>(p);
        if (w->m_map_form && w->m_map_form->WritableDataChanged())
            {
            dirty = true;
            break;
            }
        }

    if (dirty)
        {
        QMessageBox message_box(QMessageBox::Question,"Quit without saving","You have unsaved added data. Close without saving it?",
                                QMessageBox::Yes | QMessageBox::Cancel);
        QMessageBox::StandardButton button = (QMessageBox::StandardButton)message_box.exec();
        if (button == QMessageBox::Yes)
            accept = true;
        }
    else
        accept = true;

    if (accept)
        {
        // Write the geometry of the main window.
        QSettings settings;
        settings.beginGroup("mainWindow");
        settings.setValue("geometry",saveGeometry());
        settings.endGroup();

        // Write the view states of all the map windows.
        settings.beginWriteArray("mapWindows");
        int index = 0;
        for (const auto p : sub_window_list)
            {
            MapChildWindow* w = dynamic_cast<MapChildWindow*>(p);
            if (w->m_map_form)
                {
                settings.setArrayIndex(index);
                settings.setValue("geometry",w->saveGeometry());
                settings.setValue("viewState",w->m_map_form->ViewState());
                settings.setValue("metricUnits",w->m_map_form->MetricUnits());
                settings.setValue("nightMode",w->m_map_form->NightMode());
                settings.setValue("monochrome",w->m_map_form->Monochrome());
                settings.setValue("graphicsAcceleration",w->m_map_form->GraphicsAcceleration());
                settings.setValue("debugLayers",w->m_map_form->DebugLayers());
                settings.setValue("fixedLabels",w->m_map_form->FixedLabels());
                settings.setValue("draw3DBuildings",w->m_map_form->Draw3DBuildings());
                settings.setValue("drawLegend",w->m_map_form->DrawLegendEnabled());
                settings.setValue("drawScale",w->m_map_form->DrawScaleEnabled());
                settings.setValue("drawRotator",w->m_map_form->DrawRotatorEnabled());
                settings.setValue("drawRange",w->m_map_form->DrawRangeEnabled());

                settings.setValue("fileName",w->m_map_form->FileName(0)); // for backward compatibility
                settings.beginWriteArray("files");
                for (size_t i = 0; i < w->m_map_form->FileNameCount(); i++)
                    {
                    settings.setArrayIndex(int(i));
                    settings.setValue("name",w->m_map_form->FileName(i));
                    }
                settings.endArray();
                index++;
                }
            }
        settings.endArray();

        aEvent->accept();
        }
    else
        aEvent->ignore();
    }

void MainWindow::OpenRecentFileTriggered()
    {
    QAction *action = qobject_cast<QAction*>(sender());
    if (action)
        LoadMap({ action->data().toString() });
    }

void MainWindow::SetRouteProfileTriggered()
    {
    QAction *action = qobject_cast<QAction*>(sender());
    size_t index = 0;
    bool found = false;
    for (auto p : m_route_profile_action)
        {
        if (action == p)
            {
            found = true;
            break;
            }
        index++;
        }
    if (found)
        m_map_form->SetRouteProfileIndex(index);
    }

void MainWindow::on_actionOpen_triggered()
    {
    // Open a map in a new child window.
    QString map_filename = QFileDialog::getOpenFileName(this,"Open a map in a new window","","CartoType maps (*.ctm1)");
    if (map_filename.length())
        LoadMap({ map_filename });
    }

void MainWindow::on_actionOpen_in_Current_Map_triggered()
    {
    assert(m_map_form);

    // Open a map in the current map (side by side or overlaid).
    QString map_filename = QFileDialog::getOpenFileName(this,"Open a map in the current map (side by side or overlaid)","","CartoType maps (*.ctm1)");
    if (map_filename.length())
        m_map_form->LoadMap(map_filename);
    }

void MainWindow::on_actionOpen_Online_Map_triggered()
    {
    // Open an online map in a new child window.
    OnlineMapDialog dialog;
    if (dialog.exec() == QDialog::Accepted && dialog.ui->url->text().length())
        {
        auto url_and_key = dialog.ui->url->text();
        /*
        Append the api key to the URL, separated by a comma. Although commas are legal
        in URLs, and also in filenames in some OSs, they are rare and we need not support their use.
        */
        url_and_key += ",";
        url_and_key += dialog.ui->api_key->text();
        LoadMap({ url_and_key });
        }
    }

void MainWindow::LoadMap(const std::vector<QString>& aPathArray,
                         const QByteArray* aWindowGeometry,
                         const QString* aViewState,
                         const MapSettings* aMapSettings)
    {
    auto SetWindowState = [aWindowGeometry,aViewState](MapChildWindow* aWindow)
        {
        if (aWindowGeometry)
            aWindow->restoreGeometry(*aWindowGeometry);
        if (aViewState)
            {
            CartoType::ViewState v;
            auto error = v.ReadFromXml(aViewState->toStdString().c_str());
            if (!error)
                aWindow->SetView(v);
            }

        // The following code fixes a bug on Qt on Linux causing maximized windows to be restored in the wrong place.
        if (aWindow->isMaximized())
            {
            aWindow->showNormal();
            aWindow->showMaximized();
            }
        };

    // If this map has already been loaded, just bring it to the front.
    QList<QMdiSubWindow*> sub_window_list = m_ui->mdiArea->subWindowList();
    for (auto p : sub_window_list)
        {
        MapChildWindow* w = dynamic_cast<MapChildWindow*>(p);
        if (w->m_map_form && w->m_map_form->FileName(0) == aPathArray[0])
            {
            w->raise();
            w->showMaximized();
            SetWindowState(w);
            return;
            }
        }

    MapChildWindow* w = new MapChildWindow(m_ui->mdiArea,
                                           *this,
                                           aPathArray,
                                           aMapSettings ? *aMapSettings : MapSettings());
    w->setAttribute(Qt::WA_DeleteOnClose);
    if (w->m_map_form == nullptr || !w->m_map_form->Valid())
        w->close();
    else
        {
        w->showMaximized();
        SetWindowState(w);

        QSettings settings;
        QStringList files = settings.value("recentFileList").toStringList();
        files.removeAll(aPathArray[0]);
        files.prepend(aPathArray[0]);
        while (files.size() > KMaxRecentFiles)
            files.removeLast();
        settings.setValue("recentFileList",files);
        settings.sync();

        UpdateRecentFiles();
        UpdateRouteProfileMenuItems();
        }
    }

void MainWindow::OnMapFormDestroyed(MapForm* /*aMapForm*/)
    {
    }

void MainWindow::on_actionAbout_CartoType_Maps_triggered()
    {
    QString text = "The <b>CartoType Maps</b> app allows you to view maps, calculate routes, find places, and more.<br/><br/>"
                   "See <a href='https://www.cartotype.com'>cartotype.com</a> for information about creating maps, using the API, and licensing CartoType for your application.<br/><br/>"
                   "Application created using CartoType ";
    text += QString(CartoType::Version()) + "." + CartoType::Build() + ".";

    if (m_map_form)
        {
        const auto& framework = m_map_form->Framework();
        std::string s = "<br/><br/>";
        bool metadata_found = 0;
        int maps_loaded = 0;
        for (size_t i = 0; i < framework.MapCount(); i++)
            {
            auto m = framework.MapMetaData(i);
            if (m)
                maps_loaded++;
            if (m && !metadata_found)
                {
                metadata_found = true;
                char buffer[256];
                s += "Map ";
                if (m->CartoTypeVersion.Major)
                    {
                    snprintf(buffer,sizeof(buffer),"%d.%d.%d",m->CartoTypeVersion.Major,m->CartoTypeVersion.Minor,m->CartoTypeBuild);
                    s += m->DataSetName + " created by CartoType " + buffer;
                    }
                else
                    s += m->DataSetName;
                if (m->FileVersion.Major)
                    {
                    snprintf(buffer,sizeof(buffer),"%d.%d",m->FileVersion.Major,m->FileVersion.Minor);
                    s += "<br/>CTM1 version ";
                    s += buffer;
                    }
                s += ".<br/>Projection: " + m->ProjectionName;
                snprintf(buffer,sizeof(buffer),".<br/>Extent in degrees: %g, %g, %g, %g.<br/>Route table: ",m->ExtentInDegrees.MinX(),m->ExtentInDegrees.MinY(),m->ExtentInDegrees.MaxX(),m->ExtentInDegrees.MaxY());
                s += buffer;
                const char* r = "none";
                switch (m->RouteTableType)
                    {
                    case CartoType::MapTableType::KRouteTableAStar: r = "a-star (/route=a)"; break;
                    case CartoType::MapTableType::KRouteTableCH: r = "contraction hierarchy (/route=c)"; break;
                    case CartoType::MapTableType::KRouteTableTurnExpanded: r = "turn-expanded (/route=t)"; break;
                    case CartoType::MapTableType::KRouteTableCHStandAlone: r = "contraction hierarchy stand-alone (/route=cs)"; break;
                    case CartoType::MapTableType::KRouteTableTECH: r = "turn-expanded contraction hierarchy (/route=tech)"; break;
                    case CartoType::MapTableType::KRouteTableCHTiled: r = "contraction hierarchy, tiled (/route=ct)"; break;
                    case CartoType::MapTableType::KRouteTableTECHTiled: r = "turn-expanded contraction hierarchy, tiled (/route=tt)"; break;
                    case CartoType::MapTableType::KRouteTableTurnExpandedCompact: r = "turn-expanded, compact (/route=tc)"; break;
                    default: break;
                    }
                s += r;
                s += ".";
                }
            }

        text += QString(s.c_str());
        if (maps_loaded > 1)
            {
            char buffer[64];
            snprintf(buffer,sizeof(buffer),"<br/>%d other map%s loaded side by side.",maps_loaded - 1,maps_loaded > 2 ? "s" : "");
            text += QString(buffer);
            }
        }

    QMessageBox::about(this,"About CartoType Maps",text);
    }

CartoType::String MainWindow::FontPath(const char* aFontName) const
    {
    QString font_path(m_app_path + aFontName);
    if (!QFileInfo::exists(font_path))
        {
        font_path = m_app_path + "font/" + aFontName;
        if (!QFileInfo::exists(font_path))
            font_path = m_cartotype_source_path + "font/" + aFontName;
        }
    assert(sizeof(QChar) == sizeof(uint16_t));
    CartoType::String path;
    path.Set((const uint16_t*)font_path.constData(),font_path.length());
    return path;
    }

void MainWindow::ShowError(const char* aMessage,int aErrorCode)
    {
    char buffer[1024];
    if (aErrorCode)
        {
        if (aErrorCode < CartoType::KStandardErrorCodeCount)
            snprintf(buffer,sizeof(buffer),"%s: %s (error code %d)",aMessage,CartoType::ErrorString(aErrorCode).c_str(),aErrorCode);
        else
            snprintf(buffer,sizeof(buffer),"%s: %s",aMessage,CartoType::ErrorString(aErrorCode).c_str());
        QString text(buffer);
        QMessageBox::critical(this,"CartoType error",text);
        }
    else
        {
        QString text(aMessage);
        QMessageBox::information(this,"Information",text);
    }
}

void MainWindow::on_mdiArea_subWindowActivated(QMdiSubWindow* aSubWindow)
    {
    MapChildWindow* p = dynamic_cast<MapChildWindow*>(aSubWindow);

    if (p)
        m_map_form = p->m_map_form;
    else
        m_map_form = nullptr;

    EnableMenuItems();
    }

void MainWindow::on_actionScale_changed()
    {
    if (m_map_form)
        m_map_form->EnableDrawScale(m_ui->actionScale->isChecked());
    }

void MainWindow::on_actionLegend_changed()
    {
    if (m_map_form)
        m_map_form->EnableDrawLegend(m_ui->actionLegend->isChecked());
    }

void MainWindow::on_actionFind_triggered()
    {
    if (m_map_form)
        m_map_form->Find();
    }

void MainWindow::on_actionFind_Next_triggered()
    {
    if (m_map_form)
        m_map_form->FindNext();
    }

void MainWindow::EnableMenuItems()
    {
    if (m_map_form)
        {
        menuBar()->insertAction(m_ui->menuHelp->menuAction(),m_ui->menuFind->menuAction());
        menuBar()->insertAction(m_ui->menuHelp->menuAction(),m_ui->menuView->menuAction());
        menuBar()->insertAction(m_ui->menuHelp->menuAction(),m_ui->menuRoute->menuAction());
        menuBar()->insertAction(m_ui->menuHelp->menuAction(),m_ui->menuStyle->menuAction());
        m_ui->menuFile->insertAction(m_ui->actionQuit,m_ui->actionOpen_in_Current_Map);
        m_ui->menuFile->insertAction(m_recent_file_separator,m_ui->actionPrint);
        m_ui->menuFile->insertAction(m_recent_file_separator,m_ui->actionPrint_Preview);
        m_ui->menuFile->insertAction(m_recent_file_separator,m_ui->actionSave_Image_as_PNG);
        m_ui->menuFile->insertAction(m_recent_file_separator,m_ui->actionSave_Added_Data_as_CTMS);
        m_ui->menuFile->insertAction(m_recent_file_separator,m_ui->actionImport_Data_from_CTMS);
        m_ui->menuFile->insertAction(m_recent_file_separator,m_ui->actionImport_Data_from_GPX_file);
        m_ui->actionScale->setChecked(m_map_form->DrawScaleEnabled());
        m_ui->actionRotator->setChecked(m_map_form->DrawRotatorEnabled());
        m_ui->actionLegend->setChecked(m_map_form->DrawLegendEnabled());
        m_ui->actionRange->setChecked(m_map_form->DrawRangeEnabled());
        m_ui->actionPerspective_View->setChecked(m_map_form->Perspective());
        m_ui->actionEnable_debug_layers->setChecked(m_map_form->DebugLayers());
        m_ui->actionFixed_Labels->setChecked(m_map_form->FixedLabels());
        m_ui->action3D_Buildings->setChecked(m_map_form->Draw3DBuildings());
        m_ui->actionMetric_Units->setChecked(m_map_form->MetricUnits());
        m_ui->actionNight_Mode->setChecked(m_map_form->NightMode());
        m_ui->actionMonochrome->setChecked(m_map_form->Monochrome());
        m_ui->actionTurn_expanded_router->setChecked(m_map_form->PreferredRouterType() == CartoType::RouterType::TurnExpandedAStar);
        UpdateSaveAddedData();
        UpdateFindNext();
        UpdateNorthUp();
        UpdateManageRoute();
        UpdateDeletePushpins();
        UpdateGoToGridRef();
        UpdateStyleSheet();
        UpdateRouteProfileMenuItems();
        UpdateGraphicsAcceleration();
        }
    else
        {
        menuBar()->removeAction(m_ui->menuFind->menuAction());
        menuBar()->removeAction(m_ui->menuView->menuAction());
        menuBar()->removeAction(m_ui->menuRoute->menuAction());
        menuBar()->removeAction(m_ui->menuStyle->menuAction());
        m_ui->menuFile->removeAction(m_ui->actionOpen_in_Current_Map);
        m_ui->menuFile->removeAction(m_ui->actionPrint);
        m_ui->menuFile->removeAction(m_ui->actionPrint_Preview);
        m_ui->menuFile->removeAction(m_ui->actionSave_Image_as_PNG);
        m_ui->menuFile->removeAction(m_ui->actionSave_Added_Data_as_CTMS);
        m_ui->menuFile->removeAction(m_ui->actionImport_Data_from_CTMS);
        m_ui->menuFile->removeAction(m_ui->actionImport_Data_from_GPX_file);
        }
    }

void MainWindow::UpdateRecentFiles(bool aRemoveNonExistentFiles)
    {
    QSettings settings;
    QStringList files = settings.value("recentFileList").toStringList();
    int file_action_index = 0;
    int file_index = 0;
    while (file_index < files.size())
        {
        QFileInfo file_info(files[file_index]);
        if (aRemoveNonExistentFiles && !file_info.exists())
            {
            files.erase(files.begin() + file_index);
            continue;
            }

        if (file_action_index < KMaxRecentFiles)
            {
            QString text;
            text.setNum(file_action_index + 1);
            text += " ";
            text += file_info.fileName();

            // Remove the API key if any from the end of the filename shown in the menu.
            auto n = text.lastIndexOf(',');
            if (n > 0)
                text.remove(n,text.length() - n);

            m_recent_file_action[file_action_index]->setText(text);
            m_recent_file_action[file_action_index]->setData(files[file_index]);
            m_recent_file_action[file_action_index]->setVisible(true);
            file_action_index++;
            }

        file_index++;
        }

    while (file_action_index < KMaxRecentFiles)
        m_recent_file_action[file_action_index++]->setVisible(false);

    if (aRemoveNonExistentFiles)
        {
        settings.setValue("recentFileList",files);
        settings.sync();
        }
    }

void MainWindow::on_actionRotator_changed()
    {
    if (m_map_form)
        m_map_form->EnableDrawRotator(m_ui->actionRotator->isChecked());
    }

void MainWindow::on_actionRange_changed()
    {
    if (m_map_form)
        m_map_form->EnableDrawRange(m_ui->actionRange->isChecked());
    }

void MainWindow::on_actionNorth_Up_triggered()
    {
    if (m_map_form)
        m_map_form->SetRotation(0);
    }

void MainWindow::UpdateSaveAddedData()
    {
    if (m_map_form)
        m_ui->actionSave_Added_Data_as_CTMS->setEnabled(m_map_form->HasWritableData());
    }

void MainWindow::UpdateFindNext()
    {
    if (m_map_form)
        m_ui->actionFind_Next->setEnabled(m_map_form->FoundItemCount() > 1);
    }

void MainWindow::UpdateNorthUp()
    {
    if (m_map_form)
        m_ui->actionNorth_Up->setDisabled(m_map_form->Rotation() == 0);
    }

void MainWindow::UpdatePerspective()
    {
    if (m_map_form)
        m_ui->actionPerspective_View->setChecked(m_map_form->Perspective());
    }

void MainWindow::on_actionDelete_Route_triggered()
    {
    if (m_map_form)
        m_map_form->DeleteRoute();
    }

void MainWindow::on_actionDelete_Pushpins_triggered()
    {
    if (m_map_form)
        m_map_form->DeletePushpins();
    }

void MainWindow::UpdateManageRoute()
    {
    bool has_route = m_map_form && m_map_form->HasRoute();
    m_ui->actionReverse_Route->setEnabled(has_route);
    m_ui->actionDelete_Route->setEnabled(has_route);
    m_ui->actionSave_Route->setEnabled(has_route);
    m_ui->actionSave_Route_Instructions->setEnabled(has_route);
    m_ui->actionSave_Route_as_GPX->setEnabled(has_route);
    m_ui->actionView_Route_Instructions->setEnabled(has_route);
    m_ui->actionSimulate_Routing->setEnabled(has_route);
    m_ui->actionSimulate_Routing->setChecked(m_map_form->RoutingIsSimulated());
    }

void MainWindow::UpdateDeletePushpins()
    {
    bool has_pushpins = m_map_form && m_map_form->HasPushpins();
    m_ui->actionDelete_Pushpins->setEnabled(has_pushpins);
    }

void MainWindow::UpdateRouteProfile()
    {
    if (m_map_form)
        {
        size_t index = m_map_form->RouteProfileIndex();
        for (size_t i = 0; i < m_route_profile_action.size(); i++)
            m_route_profile_action[i]->setChecked(i == index);
        }
    }

void MainWindow::UpdateGoToGridRef()
    {
    if (m_map_form)
        m_ui->actionGo_to_Ordnance_Survey_grid_reference->setEnabled(m_map_form->MapIncludesGreatBritain());
    }

void MainWindow::UpdateStyleSheet()
    {
    if (m_map_form)
        {
        bool custom = m_map_form->UsingCustomStyleSheet();
        m_ui->actionUse_Custom_Style_Sheet->setChecked(custom);
        m_ui->actionChoose_Style_Sheet->setEnabled(!custom);
        m_ui->actionReload_Style_Sheet->setEnabled(!custom);
        m_ui->actionEdit_Custom_Style_Sheet->setEnabled(!m_style_dialog_open);
        }
    }

void MainWindow::UpdateGraphicsAcceleration()
    {
    if (m_map_form)
        m_ui->actionGraphics_Acceleration->setChecked(m_map_form->GraphicsAcceleration());
    }

void MainWindow::on_actionTurn_expanded_router_triggered()
    {
    if (m_map_form)
        {
        if (m_ui->actionTurn_expanded_router->isChecked())
            m_map_form->SetPreferredRouterType(CartoType::RouterType::TurnExpandedAStar);
        else
            m_map_form->SetPreferredRouterType(CartoType::RouterType::Default);
        }
    }

void MainWindow::on_actionSave_Route_Instructions_triggered()
    {
    if (m_map_form)
        m_map_form->SaveRouteInstructions();
    }

void MainWindow::on_actionSave_Route_triggered()
    {
    if (m_map_form)
        m_map_form->SaveRouteAsXml();
    }

void MainWindow::on_actionSave_Route_as_GPX_triggered()
    {
    if (m_map_form)
        m_map_form->SaveRouteAsGpx();
    }

void MainWindow::on_actionLoad_Route_triggered()
    {
    if (m_map_form)
        {
        m_map_form->LoadRouteFromXml();
        UpdateManageRoute();
        }
    }

void MainWindow::on_actionSave_Image_as_PNG_triggered()
    {
    if (m_map_form)
        m_map_form->SaveImageAsPng();
    }

void MainWindow::on_actionFind_Address_triggered()
    {
    if (m_map_form)
        m_map_form->FindAddress();
    }

void MainWindow::on_actionSave_Added_Data_as_CTMS_triggered()
    {
    if (m_map_form)
        m_map_form->SaveWritableDataAsCtms();
    }

void MainWindow::on_actionImport_Data_from_CTMS_triggered()
    {
    if (m_map_form)
        m_map_form->LoadWritableDataFromCtms();
    }

void MainWindow::on_actionImport_Data_from_GPX_file_triggered()
    {
    if (m_map_form)
        m_map_form->LoadWritableDataFromGpx();
    }

void MainWindow::on_actionSet_Scale_triggered()
    {
    if (m_map_form)
        m_map_form->SetScale();
    }

void MainWindow::on_actionGo_to_Location_triggered()
    {
    if (m_map_form)
        m_map_form->GoToLocation();
    }

void MainWindow::on_actionGo_to_Ordnance_Survey_grid_reference_triggered()
    {
    if (m_map_form)
        m_map_form->GoToOrdnanceSurveyGridRef();
    }

void MainWindow::on_actionChoose_Style_Sheet_triggered()
    {
    if (m_map_form)
        m_map_form->ChooseStyleSheet();
    }

void MainWindow::on_actionReload_Style_Sheet_triggered()
    {
    if (m_map_form)
        m_map_form->ReloadStyleSheet();
    }

void MainWindow::on_actionEdit_Custom_Style_Sheet_triggered()
    {
    if (!m_map_form)
        return;

    if (m_custom_style_sheet.empty())
        m_custom_style_sheet = m_map_form->Framework().StyleSheetText(0);
    std::string saved_custom_style_sheet = m_custom_style_sheet;

    m_style_dialog_map_form = m_map_form;
    CartoType::Result error;
    auto legend = CartoType::Legend::New(error,m_map_form->Framework());
    if (error)
        {
        ShowError("could not create Legend object",error);
        return;
        }
    m_style_dialog = std::make_unique<StyleDialog>(this,m_map_form->Framework(),saved_custom_style_sheet,std::move(legend));

    if (m_style_dialog->HaveError())
        {
        std::string s { "style sheet parse error: " };
        s += m_style_dialog->ErrorMessage() + " at '" + m_style_dialog->ErrorLocation() + "'";
        ShowError(s.c_str(),0);
        m_style_dialog = nullptr;
        return;
        }

    connect(m_style_dialog.get(),SIGNAL(ApplyStyleSheet(std::string)),m_map_form,SLOT(ApplyStyleSheet(std::string)));
    connect(m_style_dialog.get(),SIGNAL(finished(int)),this,SLOT(FinishEditingCustomStyleSheet(int)));
    m_style_dialog->show();

    // Disable edit-custom-style-sheet menu option.
    m_style_dialog_open = true;
    UpdateStyleSheet();
    }

void MainWindow::FinishEditingCustomStyleSheet(int aResult)
    {
    if (aResult == QDialog::DialogCode::Accepted)
        m_style_dialog_map_form->ApplyStyleSheet(m_style_dialog->EditedStyleSheetText());
    else
        m_style_dialog_map_form->ApplyStyleSheet(m_style_dialog->OriginalStyleSheetText());
    m_style_dialog_map_form->FinishEditingCustomStyleSheet(aResult);
    m_style_dialog_open = false;
    m_style_dialog_map_form = nullptr;

    /*
    NOTE: The style dialog (m_style_dialog) is not deleted at this point because
    other QDialog functions are called after the 'finished' signal is sent.
    The style dialog is deleted and replaced with a new one when the user
    selects 'edit custom style sheet' in the menu and
    on_actionEdit_Custom_Style_Sheet_triggered() is called.
    */

    // Enable edit-custom-style-sheet menu option.
    UpdateStyleSheet();
    }

void MainWindow::on_actionLayers_triggered()
    {
    if (m_map_form)
        m_map_form->ChooseLayers();
    }

void MainWindow::on_actionPerspective_View_changed()
    {
    if (m_map_form)
        m_map_form->SetPerspective(m_ui->actionPerspective_View->isChecked());
    }

void MainWindow::on_actionMetric_Units_changed()
    {
    if (m_map_form)
        m_map_form->SetMetricUnits(m_ui->actionMetric_Units->isChecked());
    }

void MainWindow::on_actionGraphics_Acceleration_changed()
    {
    if (m_map_form)
        m_map_form->SetGraphicsAcceleration(m_ui->actionGraphics_Acceleration->isChecked());
    }

void MainWindow::on_action3D_Buildings_changed()
    {
    if (m_map_form)
        m_map_form->SetDraw3DBuildings(m_ui->action3D_Buildings->isChecked());
    }

void MainWindow::on_actionView_Route_Instructions_triggered()
    {
    if (m_map_form)
        m_map_form->ViewRouteInstructions();
    }

void MainWindow::UpdateRouteProfileMenuItems()
    {
    if (m_map_form)
        {
        // Remove previous route profile menu items.
        for (auto p: m_route_profile_action)
            m_ui->menuRoute->removeAction(p);
        m_route_profile_action.clear();
        if (m_route_profile_separator_after)
            m_ui->menuRoute->removeAction(m_route_profile_separator_after);

        // Add route profile actions for the current map.
        size_t n = m_map_form->BuiltInProfileCount() ? m_map_form->BuiltInProfileCount() : 6;
        for (size_t i = 0; i < n; i++)
            {
            auto a = new QAction(this);
            a->setCheckable(true);
            m_route_profile_action.push_back(a);
            connect(a,SIGNAL(triggered()),this,SLOT(SetRouteProfileTriggered()));
            }
        if (m_map_form->BuiltInProfileCount())
            {
            for (size_t i = 0; i < n; i++)
                {
                CartoType::String s = m_map_form->BuiltInProfile(i)->Name;
                s.SetCase(CartoType::LetterCase::Title);
                s += " (built-in)";
                m_route_profile_action[i]->setText(s.CreateUtf8String().c_str());
                }
            }
        else
            {
            m_route_profile_action[0]->setText("Drive");
            m_route_profile_action[1]->setText("Cycle");
            m_route_profile_action[2]->setText("Walk");
            m_route_profile_action[3]->setText("Hike");
            m_route_profile_action[4]->setText("Ski");
            m_route_profile_action[5]->setText("Custom Route Profile");
            }

        m_route_profile_separator_after = m_ui->menuRoute->insertSeparator(m_ui->actionView_Route_Instructions);
        for (auto p : m_route_profile_action)
            m_ui->menuRoute->insertAction(m_route_profile_separator_after,p);
        UpdateRouteProfile();
        }
    }

void MainWindow::UpdateCustomRouteProfile()
    {
    QSettings settings;
    CartoType::MemoryOutputStream output;
    m_custom_route_profile.WriteAsXml(output);
    QString s { QString::fromUtf8((const char*)output.Data(),(int)output.Length()) };
    settings.setValue("customRouteProfile",s);
    }

void MainWindow::SetCustomStyleSheet(const std::string& aStyleSheet)
    {
    m_custom_style_sheet = aStyleSheet;
    QSettings settings;
    QString s { m_custom_style_sheet.c_str() };
    settings.setValue("customStyleSheet",s);
    }

void MainWindow::on_actionEdit_Custom_Route_Profile_triggered()
    {
    if (m_map_form && m_map_form->EditCustomRouteProfile(m_custom_route_profile))
        UpdateCustomRouteProfile();
    }

void MainWindow::on_actionSave_Custom_Route_Profile_triggered()
    {
    std::string path { GetSaveFile(*this,"Save Route Profile","CartoType route profiles","ctprofile") };
    if (path.empty())
        return;
    CartoType::MemoryOutputStream output;
    CartoType::Result error = m_custom_route_profile.WriteAsXml(output);
    if (!error)
        {
        std::ofstream f(path.c_str());
        if (f.good())
            {
            f.write((const char*)output.Data(),output.Length());
            if (!f.good())
                error = CartoType::KErrorIo;
            }
        else
            error = CartoType::KErrorIo;
        }
    if (error)
        ShowError("failed to save the route to a file",error);
    }

void MainWindow::on_actionLoad_Custom_Route_Profile_triggered()
    {
    QString path = QFileDialog::getOpenFileName(this,"Load Route Profile","","CartoType route files (*.ctprofile)");
    if (!path.length())
        return;
    CartoType::Result error = 0;
    CartoType::String filename; filename.Set(path.utf16());
    auto input = CartoType::FileInputStream::New(error,filename);
    if (!error)
        {
        CartoType::RouteProfile profile;
        error = profile.ReadFromXml(*input);
        if (!error)
            {
            m_custom_route_profile = profile;
            UpdateCustomRouteProfile();
            if (m_map_form)
                m_map_form->OnCustomRouteProfileLoaded();
            }
        }
    if (error)
        ShowError("could not load route profile",error);
    }

void MainWindow::on_actionUse_Custom_Style_Sheet_changed()
    {
    if (m_map_form)
        m_map_form->SetUseCustomStyleSheet(m_ui->actionUse_Custom_Style_Sheet->isChecked());
    }

void MainWindow::on_actionSave_Custom_Style_Sheet_triggered()
    {
    std::string path { GetSaveFile(*this,"Save Custom Style Sheet","CartoType style sheets","ctstyle") };
    if (path.empty())
        return;
    CartoType::Result error = 0;
    std::ofstream f(path.c_str());
    if (f.good())
        {
        f.write(m_custom_style_sheet.data(),m_custom_style_sheet.length());
        if (!f.good())
            error = CartoType::KErrorIo;
        }
    else
        error = CartoType::KErrorIo;
    if (error)
        ShowError("failed to save the style sheet to a file",error);
    }

void MainWindow::on_actionLoad_Custom_Style_Sheet_triggered()
    {
    QString path = QFileDialog::getOpenFileName(this,"Load Custom Style Sheet","","CartoType style sheets (*.ctstyle *.xml)");
    if (!path.length())
        return;
    CartoType::Result error = 0;
    CartoType::String filename; filename.Set(path.utf16());
    auto input = CartoType::FileInputStream::New(error,filename);
    if (!error) try
        {
        std::string buffer;
        while (!input->EndOfData())
            {
            const uint8_t* p = nullptr;
            size_t length = 0;
            input->Read(p,length);
            if (error || !p || length == 0)
                break;
            buffer.append((const char*)p,length);
            }
        SetCustomStyleSheet(buffer);
        if (m_map_form)
            m_map_form->OnCustomStyleSheetLoaded();
        } catch (CartoType::Result e) { error = e; }
    if (error)
        ShowError("could not load style sheet",error);
    }

void MainWindow::on_actionReverse_Route_triggered()
    {
    if (m_map_form)
        m_map_form->ReverseRoute();
    }

void MainWindow::on_actionPrint_triggered()
    {
    if (m_map_form)
        m_map_form->Print(false);
    }

void MainWindow::on_actionPrint_Preview_triggered()
    {
    if (m_map_form)
        m_map_form->Print(true);
    }

void MainWindow::on_actionNight_Mode_triggered(bool aChecked)
    {
    if (m_map_form)
        {
        m_map_form->SetNightMode(aChecked);
        m_ui->actionMonochrome->setChecked(m_map_form->Monochrome());
        }
    }

void MainWindow::on_actionMonochrome_triggered(bool aChecked)
    {
    if (m_map_form)
        {
        m_map_form->SetMonochrome(aChecked);
        m_ui->actionNight_Mode->setChecked(m_map_form->NightMode());
        }
    }

void MainWindow::on_actionSimulate_Routing_triggered(bool aChecked)
    {
    if (m_map_form)
        m_map_form->SetSimulateRouting(aChecked);
    }

void MainWindow::on_actionEnable_debug_layers_changed()
    {
    if (m_map_form)
        m_map_form->EnableDebugLayers(m_ui->actionEnable_debug_layers->isChecked());
    }

void MainWindow::on_actionFixed_Labels_changed()
    {
    if (m_map_form)
        m_map_form->EnableFixedLabels(m_ui->actionFixed_Labels->isChecked());
    }


