/*
CartoTypeFramework.h
Copyright 2013-2025 CartoType Ltd.
See www.cartotype.com for more information.
*/

#import <CartoType/CartoTypeBase.h>
#import <CartoType/CartoTypeRouter.h>
#import <CartoType/CartoTypeResultCode.h>
#import <CartoType/CartoTypeFrameworkParam.h>
#import <CartoType/CartoTypeLegend.h>

/**
\file
The public interface of CtFramework.
*/

/**
\mainpage CartoType

\section introduction Introduction

CartoType is a portable library for map rendering, routing, navigation and geocoding.
It uses maps created by the makemap tool supplied with the package.

\section iosapi The iOS API

The CtFramework interface is CartoType's main iOS and watchOS class.
By creating a CtFramework object you gain access to
maps, panning and zooming, searching, routing and navigation.
*/

/** Implement this protocol to handle findAsync when searching for map objects. */
@protocol CtFindAsyncProtocol <NSObject>
    /** This function is called with the array of objects that has been found by an asynchronous find function. */
    -(void)handler:(NSMutableArray<CtMapObject*>*)aObjectArray;
@end
@protocol CartoTypeFindAsyncProtocol <CtFindAsyncProtocol>
@end

/** Implement this protocol to handle findAsync when searching for map object groups. */
@protocol CtFindAsyncGroupProtocol <NSObject>
    /** This function is called with the array of object groups that has been found by an asynchronous find function. */
    -(void)handler:(NSMutableArray<CtMapObjectGroup*>*)aObjectGroupArray;
@end
@protocol CartoTypeFindAsyncGroupProtocol <CtFindAsyncGroupProtocol>
@end

/** Implement this protocol to handle an asynchronous routing function. */
@protocol CtCreateRouteAsyncProtocol <NSObject>
    /** This function is called when an asynchronous routing function has created a route. If no route could be created, aRoute is null. */
    -(void)handler:(CtResult)aResult route:(CtRoute*)aRoute;
@end
@protocol CartoTypeCreateRouteAsyncProtocol <CtCreateRouteAsyncProtocol>
@end

/** Implement this protocol to receive the result of an asynchronous map loading operation. */
@protocol CtLoadMapAsyncProtocol <NSObject>
    /** This function is called when loadMapAsync has attempted to load a map. */
    -(void)handler:(CtResult)aResult filename:(NSString*)aFileName key:(NSString*)aKey;
@end
@protocol CartoTypeLoadMapAsyncProtocol <CtLoadMapAsyncProtocol>
@end

/**
A framework for the CartoType map drawing library,
encapsulating an engine, map view, style sheet, fonts
and map data sources.
*/
@interface CtFramework: NSObject

/**
Initializes a new framework object.

aMapFileName is the file name of the main map.
aStyleSheetFileName is the name of the style sheet file.
aFontFileName is the name of the first font to be loaded. Other fonts may be loaded at any time.
aWidth and aHeight are the width and height of the map in pixels.

Map, style sheet and font file names may be names of resources in the app's resource bundle, with no extension, or ordinary file
paths with an extension. When loading resources from bundles, the extensions '.ctm1',
'.xml', and '.ttf' are assumed for maps, style sheets and fonts respectively.
*/
-(id)initWithMap:(NSString*)aMapFileName andStyle:(NSString*)aStyleSheetFileName andFont:(NSString*)aFontFileName andWidth:(int)aWidth andHeight:(int)aHeight;

/**
Initializes a new framework object from a parameter object.
*/
-(id)initWithParam:(CtFrameworkParam*)aParam;

/** Returns the last error code set by a CtFramework method that does not return an error code. */
-(CtResult)lastError;

/**
Returns a pointer to the underlying C++ CartoType::Framework object.
*/
-(void*)framework;

/** Creates a copy of the framework with the same fonts, maps and style sheets. */
-(id)copy;

/** Configures the CartoType framework by reading an XML configuration file. */
-(CtResult)configure:(NSString*)aConfigFileName;

/** Returns a Git description of the current CartoType version as a string of the form MAJOR.MINOR-BUILD-COMMIT. */
+(NSString*)description;

/** Returns the current CartoType version as a string of the form MAJOR.MINOR. */
+(NSString*)version;

/** Returns the current CartoType build as a string. */
+(NSString*)build;

/** Supplies a license key to enable the licensed version of the CartoType library. */
-(CtResult)license:(NSString*)aKey;

/**
Sets a parameter to be used by a data provider.
For example, to set the 'permanent' parameter used by the Mapbox geocoding API,
use the key "mapbox/geocoding/permanent" and the value "true".
If the value is empty the entry for the key is deleted.
*/
-(CtResult)setProviderParam:(NSString*)aKey value:(NSString*)aValue;

/** The name of the licensee, or "no licensee: for evaluation only" if there is no licensee. */
@property (nonatomic) NSString* licensee;

/** The expiry date of the license in the form YYYY-MM-DD. */
@property (nonatomic) NSString* expiryDate;

/** The date on which the application was built in the form YYYY-MM-DD. */
@property (nonatomic) NSString* appBuildDate;

/** The copyright string stored in the main map database. */
@property (nonatomic) NSString* copyright;

/** Sets the copyright notice displayed at the bottom-right corner of the map to that of the main map database. */
-(CtResult)setCopyrightNotice;

/** Sets the copyright notice displayed at the bottom-right corner of the map. */
-(CtResult)setCopyrightNotice:(NSString*)aCopyright;

/** Sets the legend. If a legend is supplied, it is used, otherwise a standard legend is created. */
-(CtResult)setLegend:(CtNoticePosition*)aPosition legend:(CtLegend*)aLegend;

/** Sets the legend to a standard legend with a specified width and position. */
-(CtResult)setLegend:(CtNoticeAnchor)aAnchor width:(double)aWidth unit:(NSString*)aUnit;

/** Enables or disables the drawing of a legend that has been supplied using setLegend. */
@property (nonatomic) bool legendEnabled;

/** Creates a scale bar.  If a legend is supplied, a scale bar is added to it and it is used, otherwise a standard scale bar is created.*/
-(CtResult)setScaleBar:(CtNoticePosition*)aPosition legend:(CtLegend*)aLegend;

/** Creates a standard scale bar at a specified position and width. */
-(CtResult)setScaleBar:(CtNoticeAnchor)aAnchor width:(double)aWidth unit:(NSString*)aUnit;

/** Enables or disables the drawing of a scale bar that has been supplied using setScaleBar. */
@property (nonatomic) bool scaleBarEnabled;

/** Creates turn instructions  If a legend is supplied, turn instructions are added to it and it is used, otherwise standard turn instructions are created. If aAbbreviate is true the turn instructions are abbreviated. */
-(CtResult)setTurnInstructions:(CtNoticePosition*)aPosition legend:(CtLegend*)aLegend abbreviate:(bool)aAbbreviate;

/** Creates standard turn instructions with a specified position and width. Optionally (if aTextSize is greater than zero) sets the text size. If aAbbreviate is true the turn instructions are abbreviated. */
-(CtResult)setTurnInstructions:(CtNoticeAnchor)aAnchor width:(double)aWidth widthUnit:(NSString*)aWidthUnit textSize:(double)aTextSize textSizeUnit:(NSString*)aTextSizeUnit abbreviate:(bool)aAbbreviate;

/** Enables or disables the drawing of a turn instruction notice that has been supplied using SetTurnInstructions. */
@property (nonatomic) bool turnInstructionsEnabled;

/**
Sets the instruction to be displayed for the current turn, overriding automatically generated instructions.
To revert to automatically generated instructions, call this function with empty or null text.
This function has no effect if turn instructions have not been created using setTurnInstructions,
or have been disabled by calling enableTurnInstructions(false).
*/
-(void)setTurnInstructionText:(NSString*)aText;

/**
Returns the instruction to be displayed for the current turn.
This function returns an empty string if turn instructions have not been created using setTurnInstructions.
*/
-(NSString*)turnInstructionText;

/**
Controls whether notices (the scale bar, legend and copyright notice) are drawn automatically.
If not, they can be drawn using DrawNotices.
*/
-(void)drawNoticesAutomatically:(bool)aEnable;

/** Returns true if any notices such as a scale bar, legend or copyright notice exist. */
@property (nonatomic) bool hasNotices;

/**
Returns a bitmap containing notices (legend, scale bar and copyright notice).
Turns off automatic drawing of notices to the map bitmap if it is in effect.
*/
-(CtPositionedBitmap*)noticeBitmap;

/**
Loads a map in addition to any maps already loaded.
The map file name may be the name of a resource in the app's resource bundle, with no extension ('.ctm1' is assumed),
or an ordinary file path with an extension.
*/
-(CtResult)loadMap:(NSString*)aMapFileName;

/**
Loads a map in addition to any maps already loaded.
The map file name may be the name of a resource in the app's resource bundle, with no extension ('.ctm1' is assumed),
or an ordinary file path with an extension, or a URL.
 
An API key or encryption key may be supplied in aKey.
A file policy is supplied in aPolicy.
*/
-(CtResult)loadMap:(NSString*)aMapFileName key:(NSString*)aKey policy:(CtFilePolicy)aPolicy;

/**
Loads a map asynchronously, calling aCallback with the result of the operation.
An API key or encryption key may be supplied in aKey.
A file policy is supplied in aPolicy.
*/
-(CtResult)loadMapAsync:(id<CtLoadMapAsyncProtocol>)aCallback filename:(NSString*)aMapFileName key:(NSString*)aKey policy:(CtFilePolicy)aPolicy;

/**
Initiates automatic map loading, specifying a set of tile maps,
which must be compatible with the base map (the map loaded on construction of the Framework object).
Unloads all maps apart from the base map and the in-memory map.
*/
-(CtResult)loadMapsAutomatically:(CtTileParam*)aTileParam;

/** Cancels automatic map loading. @see loadMapsAutomatically. */
-(void)cancelAutomaticMapLoading;

/**
Unloads a map, selecting it by handle.
It is illegal to unload the main map database or the in-memory database.
*/
-(CtResult)unloadMapByHandle:(uint32_t)aHandle;

/** Create a writable (editable) map of the specified type and load it. */
-(CtResult)createWritableMapOfType:(CtWritableMapType)aType;

/**
Saves a map identified by its handle by writing it in the specified format.
Only writable map databases may be saved in this way.
Only the CTMS (CartoType Map Serialized)
and Keyhole Markup Language (KML) formats are supported.
 */
-(CtResult)saveMap:(uint32_t)aHandle fileName:(NSString*)aFileName fileType:(CtFileType)aFileType;

/**
Reads map data from the format given by aFileType and
merges it into the map identified by aHandle.
Only writable map databases may be modified in this way.
Only the CTMS (CartoType Map Serialized)
and Keyhole Markup Language (KML) formats are supported.
*/
-(CtResult)readMap:(uint32_t)aHandle fileName:(NSString*)aFileName fileType:(CtFileType)aFileType;

/**
Saves selected objects to a map identified by its handle by writing them as an array of bytes in CTMS format.
aFindParam can be used to select the objects to be saved.
If an error occurs the return value is a single-byte data object containing the error code.
*/
-(NSData*)saveMap:(uint32_t)aHandle param:(CtFindParam*)aFindParam;

/**
Reads map data from aData in CTMS format and merges it into the map identified by aHandle,
which must refer to a writable map database.
*/
-(CtResult)readMap:(uint32_t)aHandle data:(NSData*)aData;

/**
Writes a map image to a file of the specified type.
Only the PNG file type is supported.
If aCompress is true, uses lossy compression to reduce the file size.
For PNG files compression is achieved by palettization.
*/
-(CtResult)writeMapImage:(NSString*)aFileName fileType:(CtFileType)aFileType compress:(bool)aCompress;

/** Returns true if a map, identified by its handle, is empty. */
-(bool)mapIsEmpty:(uint32_t)aHandle;

/** Enables a map, selecting it by handle. */
-(void)enableMap:(uint32_t)aHandle;

/** Disables a map, selecting it by handle. */
-(void)disableMap:(uint32_t)aHandle;

/** The handle of the map most recently loaded, or created using createWritableMap. */
@property (nonatomic) uint32_t lastMapHandle;

/** The handle of the main map. */
@property (nonatomic) uint32_t mainMapHandle;

/** The handle of the in-memory map used for routes. */
@property (nonatomic) uint32_t memoryMapHandle;

/** The number of maps currently loaded, including the main map and the in-memory map database. */
@property (nonatomic) size_t mapCount;

/** Returns the metadata for a map, identified by its index, if available. Returns null if no metadata is available. Metadata is currently available only for CTM1 maps. */
-(CtMapMetaData*)mapMetaData:(uint32_t)aIndex;

/**
Loads a font in addition to any fonts already loaded.
The font file name may be the name of a resource in the app's resource bundle, with no extension ('.ttf' is assumed),
or an ordinary file path with an extension.
*/
-(CtResult)loadFont:(NSString*)aFontFileName;

/**
Sets the current style sheet by loading it from a file.
The style sheet file name may be the name of a resource in the app's resource bundle, with no extension ('.xml' is assumed),
or an ordinary file path with an extension.
*/
-(CtResult)setStyleSheet:(NSString*)aStyleSheetFileName;

/**
Reloads a sheet from the file it was originally loaded from.
This function can be used to apply changes after a style sheet is modified.
*/
-(CtResult)reloadStyleSheet;

/**
Resizes a map, adjusting the projection such that the same map area
is visible in the resized map.
*/
-(CtResult)setMapWidth:(int)aWidth andHeight:(int)aHeight;

/** The display resolution used by CartoType in dots per inch. It has to be set correctly for map scaling and lettering to work properly.  */
@property (nonatomic) double resolutionDpi;

/** The current scale denominator. Use the denominator 50000 for a scale of 1:50,000. */
@property (nonatomic) double scaleDenominator;

/** The scale denominator for the current view, adjusting for any projection distortion: for example, use the denominator 50000 for a scale of 1:50,000. */
@property (nonatomic) double scaleDenominatorInView;

/**
Calculates the scale denominator at the equator for a certain Open Street Map zoom level
(0 maps the length of the equator to aImageSizeInPixels pixels, 1 = half the length of the equator, etc.).
*/
-(double)scaleDenominatorFromZoomLevel:(double)aZoomLevel forImageSize:(int)aImageSizeInPixels;

/**
Calculates the Open Street Map zoom level (0 maps the length of the equator to aImageSizeInPixels pixels,
1 = half the length of the equator, etc.) for a certain scale denominator.
*/
-(double)zoomLevelFromScaleDenominator:(double)aScaleDenominator forImageSize:(int)aImageSizeInPixels;

/** Returns the great-circle distance in meters between two points. Uses the WGS84 ellipsoid. */
-(double)distanceInMetersFrom:(CtPoint)aStart to:(CtPoint)aEnd coordType:(CtCoordType)aCoordType;

/** True if perspective mode is on. */
@property (nonatomic) bool perspective;

/** The current perspective parameters. */
@property (nonatomic) CtPerspectiveParam* perspectiveParam;

/** True if buildings are drawn in 3D where data is available. */
@property (nonatomic) bool draw3DBuildings;

/** True if transitions are smoothly animated in graphics-accelerated drawing. */
@property (nonatomic) bool animateTransitions;

/** The number of frames per second used when drawing labels. Clamped to the range 1...30. */
@property (nonatomic) double labelFPS;

/** True if labels are drawn, as far as possible, in fixed positions, and not moved when the map is panned. */
@property (nonatomic) bool fixedLabels;

/** The up direction for labels as an angle in degrees measured clockwise from straight up. */
@property (nonatomic) double labelUpAngle;

/** Zooms in by a specified zoom factor, or zooms out if the zoom factor is less than 1. */
-(CtResult)zoom:(double)aZoomFactor;

/**
Zooms in by a specified zoom factor, or zooms out if the zoom factor is less than 1,
centering the zoom at the specified position, so that it remains at the same place on the display.
This function does not zoom in beyond the minimum
scale denominator (500) or zoom out beyond the maximum scale denominator
(1,000,000,000) or beyond a size where the map occupies the whole display.
 */
-(CtResult)zoomAt:(double)aZoomFactor x:(double)aX y:(double)aY coordType:(CtCoordType)aCoordType;

/** Rotates the map about its center by an angle in degrees. */
-(CtResult)rotate:(double)aAngle;

/** Rotates the map about a specified point by an angle given in degrees. */
-(CtResult)rotateAt:(double)aAngle x:(double)aX y:(double)aY coordType:(CtCoordType)aCoordType;

/** Sets the map's orientation to a rotation about a specified point by an absolute angle given in degrees. */
-(CtResult)setRotationAt:(double)aAngle x:(double)aX y:(double)aY coordType:(CtCoordType)aCoordType;

/** The current rotation angle of the map in degrees. */
@property (nonatomic) double rotation;

/** Pans (slides) the map by a certain distance in pixels, horizontally or vertically or both. */
-(CtResult)panX:(int)aDx andY:(int)aDy;

/** Pans (slides) the map so that the position aFrom moves to aTo. */
-(CtResult)panFrom:(CtPoint)aFrom fromCoordType:(CtCoordType)aFromCoordType to:(CtPoint)aTo toCoordType:(CtCoordType) aToCoordType;

/**
Sets the center of the map to the position aCenter, specifying the coordinate type,
which may be display pixels, map coordinates or latitude and longitude.
*/
-(CtResult)setViewCenter:(CtPoint)aCenter coordType:(CtCoordType)aCoordType;

/**
Sets the view area of the map. specifying the coordinate type,
which may be display pixels, map coordinates or latitude and longitude.
The projection is scaled and shifted so that the 
projected map will contain the specified area.

You can also specify a margin in pixels around the display area,
and a minimum scale denominator to ensure that when displaying a small map
object the map is not zoomed right in to it. 
*/
-(CtResult)setView:(CtRect)aView coordType:(CtCoordType)aCoordType margin:(int)aMarginInPixels minScale:(int)aMinScaleDenominator;

/** Sets the view to show certain geometry, with a margin in pixels, and at a minimum scale. */
-(CtResult)setView:(CtGeometry*)aGeometry margin:(int)aMarginInPixels minScale:(int)aMinScaleDenominator;

/**
Sets the view to the current route,
with a margin in pixels, and at a minimum scale.
This function does nothing if the selected route doesn't exist.
*/
-(CtResult)setViewToRoute:(int)aMarginInPixels minScale:(int)aMinScaleDenominator;

/** Shows the whole map, zooming out as far as necessary. */
-(CtResult)setViewToWholeMap;

/**
Sets the limits to zooming and panning.

Zooming is restricted to the range between aMinScaleDenominator and aMaxScaleDenominator,
and panning is restricted so that the edges of the display always touch the bounds of aPanArea.

Use the value zero to remove either or both of the two zoom limits. Use an empty (not null) geometry object to
remove panning limits. Use a null geometry object to limit panning to the extent of the map.
*/
-(CtResult)setViewLimitsToMinScale:(double)aMinScaleDenominator maxScale:(double)aMaxScaleDenominator panArea:(CtGeometry*)aPanArea;

/** Sets the view to show a single map object, with a margin in pixels, and at a minimum scale. */
-(CtResult)setViewObject:(CtMapObject*)aMapObject margin:(int)aMarginInPixels minScale:(int)aMinScaleDenominator;

/** Sets the view to show a group of map objects, with a margin in pixels, and at a minimum scale. */
-(CtResult)setViewObjects:(NSArray<CtMapObject*>*)aObjectArray margin:(int)aMarginInPixels minScale:(int)aMinScaleDenominator;

/** Gets the current view coordinates in display pixels, map coordinates or lat/long. */
-(CtResult)getView:(CtRect*)aView coordType:(CtCoordType)aCoordType;

/** The current view state. */
@property (nonatomic) CtViewState* viewState;

/** Gets the extent of the entire map in display pixels, map coordinates or lat/long. */
-(CtResult)getMapExtent:(CtRect*)aExtent coordType:(CtCoordType)aCoordType;

/** The current map projection as a proj.4 parameter string if possible. If not returns the empty string. */
@property (nonatomic) NSString* projectionAsProj4Param;

/** Inserts an object into a writable map. */
-(CtResult)insertMapObject:(CtMapObjectParam*)aParam;

/**
Inserts a pushpin: a point object belonging to the 'pushpin' layer.
The pushpin is inserted into the built-in modifiable map.
 
If aColor is not empty it is used as the _color attribute, which,
in the standard style sheet, causes the pushpin icon to be drawn in
that color instead of the standard color.
 
If aIconCharacter is not zero it is used to set the _iconText attributes,
which, in the standard style sheet, causes that character to be used
as an icon. The MapKeyIcons font is supplied with CartoType and contains
icons in the private-character range U+E002...U+E0B1.
 
If aId is zero, a new pushpin is inserted.
If aId is non-zero, an existing pushpin with the specified ID, if any exists, is replaced.
The ID of the new object is returned in the id member of the return value.
*/
-(CtIdAndResult)insertPushPin:(CtPoint)aPoint coordType:(CtCoordType)aCoordType stringAttrib:(NSString*)aStringAttrib color:(NSString*)aColor iconChar:(int)aIconCharacter id:(uint64_t)aId;

/**
Inserts an object by copying an existing object. If aRadius is greater than zero,
converts the object to an envelope using the specified radius.
Uses the supplied ID if aId is non-zero, otherwise generates an ID.
This function fails if the map identified by aMapHandle is not writable.
If the supplied ID is that of an existing object, the existing object is replaced if aReplace is true,
otherwise the function fails and the result code CtErrorDuplicate is returned.
The ID of the new object is returned in the id member of the return value.
*/
-(CtIdAndResult)insertCopyOfMapObject:(CtMapObject*)aObject map:(uint32_t)aMapHandle layerName:(NSString*)aLayerName radius:(double)aRadius radiusCoordType:(CtCoordType)aRadiusCoordType id:(uint64_t)aId replace:(bool)aReplace;

/**
Deletes map objects in the range aStartID...aEndID inclusive from a writable map identified by its handle.
If aCondition is non-null only objects satisfying the condition are deleted.
The number of objects deleted is returned.
*/
-(uint64_t)deleteObjectsFromMap:(uint32_t)aMapHandle fromID:(uint64_t)aStartID toID:(uint64_t)aEndId withCondition:(NSString*)aCondition;

/** Loads a map object identified by its map handle and object ID. Returns nil if the object is not found. */
-(CtMapObject*)loadMapObjectFromMap:(uint32_t)aMapHandle withID:(uint64_t)aObjectId;

/**
Reads route and track objects from a GPX file and inserts
them into the map identified by aMapHandle. This function fails if the map is not writable.
 
Waypoint objects (wpt) are made into points in the gpx/wpt layer.
Route objects (rte) are made into lines in the gpx/rte layer.
Track objects (trk) are made into lines in the gpx/trk layer.
*/
-(CtResult)readGpx:(uint32_t)aMapHandle file:(NSString*)aFileName;

/**
Returns the range: a polygon containing the area reachable within a certain time in seconds or distance in meters.
 
Uses the supplied route profile if it is non-null, otherwise the current profile.
If a time is supplied, the profile's bonuses and penalties are suppressed.
If a distance is supplied, shortest routes are used.
You can optionally insert the returned CtGeometry object into the map using insertMapObject.
*/
-(CtGeometry*)range:(CtRouteProfile*)aProfile point:(CtPoint)aPoint coordType:(CtCoordType)aCoordType timeOrDistance:(double)aTimeOrDistance isTime:(bool)aIsTime;

/**
Returns an object containing route times in seconds and distances in metres from aFrom to all points in aTo.
The two point sets must use the same coordinate type.
If routing fails for any pair of points the time and distance are set to UINT32_MAX.
It is legal for aFrom to refer to the same vector as aTo. Times and distances where the start and end point are the same are set to zero.
*/
-(CtTimeAndDistanceMatrix*)timeAndDistanceMatrix:(CtPointSet*)aFrom to:(CtPointSet*)aTo;

/**
Determines the mutual accessibility of a set of points for routing purposes.
This function is implemented only for the turn-expanded router.

The points are supplied in X,Y pairs in aPointArray.
If aGroupArray is non-null it is filled with the connectivity group IDs of the points.
A set of points that is mutually accessible will have identical non-zero group IDs.
A group ID of zero indicates that a point is not near enough to a road or other routable segment.
Group IDs are ordered by group sizes. Group 1 is the group with the largest number of connected route nodes,
therefore in general a point needs to be in group 1 to take part in routing.
*/
-(bool)mutuallyAccessible:(NSMutableArray<NSNumber*>*)aPointArray coordType:(CtCoordType)aCoordType groupArray:(NSMutableArray<NSNumber*>*)aGroupArray;

/** Enables the named map layer. */
-(void)enableLayer:(NSString*)aLayerName;

/** Disables the named map layer. */
-(void)disableLayer:(NSString*)aLayerName;

/** Returns true if the named map layer is enabled, false otherwise. */
-(bool)layerIsEnabled:(NSString*)aLayerName;

/** Returns the names of the layers in the map as an array of NSString objects. */
-(NSMutableArray<NSString*>*)layerNames;

/** Converts coordinates in aGeometry to another coordinate type. */
-(CtResult)convertCoords:(CtGeometry*)aGeometry to:(CtCoordType)aToCoordType;

/** Converts a point between any combination of lat/long, map coordinates and display pixels. */
-(CtPointAndResult)convertPoint:(CtPoint)aPoint from:(CtCoordType)aFromCoordType to:(CtCoordType)aToCoordType;

/** The name of the data in the main map database. */
@property (nonatomic) NSString* dataSetName;

/** Converts a distance in pixels to map meters (projected meters). */
-(double)pixelsToMeters:(double)aPixels;

/** Converts a distance in map meters (projected meters) to pixels. */
-(double)metersToPixels:(double)aMeters;

/**
Sets the map used to store editable objects. Does not affect objects already created.
If this function is not called, editable objects are stored in the map used for
route objects.
*/
-(CtResult)editSetWritableMap:(uint32_t)aMapHandle;

/** Creates a new editable line object starting at the specified point. */
-(CtResult)editNewLineObject:(CtPoint)aDisplayPoint;

/** Creates a new editable polygon object starting at the specified point. */
-(CtResult)editNewPolygonObject:(CtPoint)aDisplayPoint;

/** Moves the editable object's current point if any. */
-(CtResult)editMoveCurrentPoint:(CtPoint)aDisplayPoint;
    
/** Adds a new point to the editable object by copying the current point. */
-(CtResult)editAddCurrentPoint;

/** Deletes the editable object's current point unless that would result in a line object of fewer than 2 points, or a polygon object of fewer than 3. */
-(CtResult)editDeleteCurrentPoint;

/** Deletes the current editable object. */
-(CtResult)editDeleteCurrentObject;

/**
Selects an editable object by selecting the nearest point of any editable object within a given radius.
If the point is not within the given radius of an existing point, creates a new point.
*/
-(CtResult)editSelectNearestPoint:(CtPoint)aDisplayPoint radius:(double)aRadiusInMillimeters;

/**
Inserts the currently edited object into a chosen layer, converting it from an edited object into an ordinary object.
If aId is non-zero and aReplace is true, replaces any existing object with that ID, otherwise creates a new object.
The ID of the new object is returned as part of the result.
*/
-(CtIdAndResult)editInsertCurrentObjectInLayer:(NSString*)aLayer id:(uint64_t)aId replace:(bool)aReplace;

/** Sets a string attribute in the currently edited object. If aKey is empty, sets the label. If aValue is empty, deletes the attribute. */
-(CtResult)editSetCurrentObjectStringAttribute:(NSString*)aKey value:(NSString*)aValue;

/** Sets the feature info of the currently edited object. */
-(CtResult)editSetCurrentObjectFeatureInfo:(CtFeatureInfo*)aFeatureInfo;

/**
Returns the area and length of the currently edited object.
For line objects returns 0 and the length in meters.
For polygon objects returns the area in square meters and the perimeter in meters.
*/
-(CtAreaLengthAndResult)editGetCurrentObjectAreaAndLength;

/** Returns a bitmap containing the current map, redrawing it first if necessary. */
-(UIImage*)mapBitmap;

/**
Returns a bitmap containing a tile specified using zoom level, x and y coordinates
used as in Google Maps or OpenStreetMap.

This function works properly only if the map uses the Web Mercator projection.
*/
-(UIImage*)tileBitmapWithSize:(int)aSize andZoom:(int)aZoom andX:(int)aX andY:(int)aY;

/**
Enables or disables clipping the map background to the map bounds.

Use this when creating map tiles to be overlaid on a larger map,
so that the background (sea) color does not erase the larger map's graphics.
The return value is the previous state.
*/
-(bool)clipBackgroundToMapBounds:(bool)aEnable;

/**
Sets the number of zoom levels by which to zoom out when creating a tile bitmap
returned by tileBitmapWithSize.

The number may be 0, 1, or 2. If it is 0, each tile is self-contained. If it is 1,
each tile is cut from a larger tile containing four tiles. If it is 2, each tile
is cut from a larger tile containing sixteen tiles. The values 1 and 2 are useful
in allowing longer labels to be used when creating small tiles; if not, many
street labels will not be drawn because the street is cut in two by the tile border
and the resulting sections are too short.

The return value is the previous setting.
*/
-(int)setTileOverSizeZoomLevels:(int)aLevels;

/**
Loads an icon from a file. It will be used whenever the ID aId is referenced
in the style sheet, overriding any icon specified in the style sheet.

The icon file may be in PNG or JPEG format.
 
aHotSpot is the point, relative to the top left corner of the icon, that should be
made to coincide with the point of interest marked by the icon.

aLabelPos is the point, relative to the hot spot, at which the center
of the baseline of any label text is placed.
 
aLabelMaxLength is the maximum length of label text in pixels.
Labels too long to be drawn on the icon are drawn next to it.
*/
-(CtResult)loadIcon:(NSString*)aFileName id:(NSString*)aId hotSpot:(CtPoint)aHotSpot labelPos:(CtPoint)aLabelPos labelMaxlength:(int)aLabelMaxLength;

/**
A general search function, allowing any combination of clip rectangle,
choice of layers, choice of attributes, text and string match method, and condition.

The returned values are CtMapObject objects which are placed in the array
aObjectArray, which is cleared before the search.

aObjectArray contains CtMapObject objects.
*/
-(CtResult)find:(NSMutableArray<CtMapObject*>*)aObjectArray withParam:(CtFindParam*)aFindParam;

/**
Performs a general find operation asynchronously and calls the handler method of the object aCallBack
(from a different thread) with the map objects that have been found when the operation has been completed.
If aOverride is true, any pending asynchronous find operations that have not yet been started
are deleted. Asynchronous find operations are executed in order.
*/
-(CtResult)findAsync:(id<CtFindAsyncProtocol>)aCallback param:(CtFindParam*)aFindParam override:(bool)aOverride;

/**
A search function to make interactive searching easier. It returns map objects grouped by
names which can be used to identify the groups in a list box. For example, a search for pubs
using the word 'Red' as the search term might return one group with the name 'Red Lion' and
another group with the name 'Red Cow'.

An array of CtMapObjectGroup objects is returned, or nil if no objects are found,
or a nil or invalid find parameter object is supplied.
*/
-(NSMutableArray<CtMapObjectGroup*>*)findGroups:(CtFindParam*)aFindParam;

/**
An asynchronous search function to make interactive searching easier.

It returns map objects grouped by names which can be used to identify the groups in a list box. For example, a search for pubs
using the word 'Red' as the search term might return one group with the name 'Red Lion' and
another group with the name 'Red Cow'.

The handler method of the object aCallback is called (from a different thread) with the map object groups that have been found
when the operation has been completed. If aOverride is true, any pending asynchronous find operations that have not yet been started
are deleted. Asynchronous find operations are executed in order.
*/
-(CtResult)findAsyncGroups:(id<CtFindAsyncGroupProtocol>)aCallback param:(CtFindParam*)aFindParam override:(bool)aOverride;

/**
A function to find nearby objects.
You can also construct a CtFindParam object from the same parameters, modify them if needed, and call find:withParam.
*/
-(CtResult)findNearby:(NSMutableArray<CtMapObject*>*)aObjectArray featureType:(CtFeatureType)aFeatureType text:(NSString*)aText location:(CtGeometry*)aLocation;

/**
Finds an address using structured data. At least one of the fields in the CtAddress
object must be filled in. If aFuzzy is true, uses fuzzy matching.

The objects returned are in map coordinates. They are placed in the array
aObjectArray, which is cleared before the search.
 
aObjectArray contains CtMapObject objects.
*/
-(CtResult)find:(NSMutableArray<CtMapObject*>*)aObjectArray maxItems:(int)aMaxItems address:(CtAddress*)aAddress fuzzy:(bool)aFuzzy;

/**
Finds an address asynchronously and calls the handler method of the object aCallBack
(from a different thread) with the map objects that have been found when the operation has been completed.
If aOverride is true, any pending asynchronous find operations that have not yet been started
are deleted. Asynchronous find operations are executed in order.
*/
-(CtResult)findAddressAsync:(id<CtFindAsyncProtocol>)aCallback maxItems:(int)aMaxItems address:(CtAddress*)aAddress fuzzy:(bool)aFuzzy override:(bool)aOverride;

/**
Finds all objects within aRadius of the point aPoint on the display. The point and the radius are given in pixels.
Unlike the other find functions, this function deals with the graphic representation of objects, so it
will work correctly when the user clicks on an icon, but not on the actual location of the object represented by
the icon.
*/
-(CtResult)findInDisplay:(NSMutableArray<CtMapObject*>*)aObjectArray maxItems:(int)aMaxItems point:(CtPoint)aPoint radius:(double)aRadius;

/** Finds all objects in a named layer or list of layers (space or comma separated), within specified bounds. */
-(CtResult)findInLayer:(NSMutableArray<CtMapObject*>*)aObjectArray maxItems:(int)aMaxItems layer:(NSString*)aLayer clip:(CtRect)aClip coordType:(CtCoordType)aCoordType;

/**
Finds all polygon objects containing a certain path.
To use standard parameters pass null as aFindParam.
To set parameters other than the path and the condition of finding polygons, pass a CtFindParam object with your chosen parameters.
*/
-(CtResult)findPolygonsContainingPath:(NSMutableArray<CtMapObject*>*)aObjectArray path:(CtGeometry*)aPath findParam:(CtFindParam*)aFindParam;

/**
Finds all point objects contained in a certain closed path.
To use standard parameters pass null as aFindParam.
To set parameters other than the path and the condition of finding points, pass a CtFindParam object with your chosen parameters.
 */
-(CtResult)findPointsInPath:(NSMutableArray<CtMapObject*>*)aObjectArray path:(CtGeometry*)aPath findParam:(CtFindParam*)aFindParam;

/**
Returns a string summarizing a geocode for a map object.
The summary is a string version of the address returned by getAddress.
 
Calls to this function are limited in number in unlicensed (evaluation) versions of the CartoType library.
*/
-(NSString*)geoCodeSummary:(CtMapObject*)aMapObject;

/**
Returns a string summarizing a geocode for a geographical location.
The summary is a string version of the address returned by getAddress.
 
Calls to this function are limited in number in unlicensed (evaluation) versions of the CartoType library.
*/
-(NSString*)geoCodeSummary:(CtPoint)aPoint coordType:(CtCoordType)aCoordType;

/**
Creates an address for a map object.

Calls to this function are limited in number in unlicensed (evaluation) versions of the CartoType library.
*/
-(CtResult)getAddress:(CtAddress*)aAddress mapObject:(CartoTypeMapObject*)aMapObject;

/**
Creates an address for a map object, using attributes of the object only.
Returns CtErrorNotFound if the relevant address parts were not found.
*/
-(CtResult)getAddressFast:(CtAddress*)aAddress mapObject:(CtMapObject*)aMapObject;

/**
Creates an address for a geographical location.

Calls to this function are limited in number in unlicensed (evaluation) versions of the CartoType library.
*/
-(CtResult)getAddress:(CtAddress*)aAddress point:(CtPoint)aPoint coordType:(CtCoordType)aCoordType;

/**
Returns the height in meters at a single point, or -32768 if it is unavailable.

The layer 'terrain-height-metres' is used by preference.
It contains heights to meter precision. If that layer is not available,
the layer 'terrain-height-feet' is used, which contains heights to 100-foot precision.
*/
-(double)height:(CtPoint)aPoint coordType:(CtCoordType)aCoordType;

/**
Finds the heights above sea level of the points making up a geometry object.
Heights are returned in meters.

The layer 'terrain-height-metres' is used by preference.
It contains heights to meter precision. If that layer is not available,
the layer 'terrain-height-feet' is used, which contains heights to 100-foot precision.
 
Unknown heights are set to -32768.
*/
-(NSMutableArray<NSNumber*>*)heights:(CtGeometry*)aGeometry;

/** Creates a height profile for a route as an array of height profile points. If aIntervalinMeters is not positive, an interval of 100 meters is used. Returns an empty array if heights are not available. */
-(NSMutableArray<CtHeightProfilePoint*>*) heightProfile:(CtRoute*)aRoute intervalInMeters:(double)aIntervalInMeters;

/** Sets a variable used in conditional compilation of the style sheet to a string value. */
-(void)setStyleSheetVariable:(NSString*)aName toString:(NSString*)aValue;

/** Sets a variable used in conditional compilation of the style sheet to an integer value. */
-(void)setStyleSheetVariable:(NSString*)aName toInt:(int)aValue;

/** True if night mode is set. In night mode the map is shaded dark blue and roads are made brighter than other features. */
@property (nonatomic) bool nightMode;

/** True if monochrome mode is set. In monochrome mode the map is drawn using greys. */
@property (nonatomic) bool monochrome;

/**
The night mode color, which should be a dark color (although this is not enforced).

To set a custom blend fraction, use the alpha value. For example, set the color's alpha value to 153 to blend the night mode color with
a style color in the ratio 0.6 : 0.4 (153/255 = 0.6). Use an alpha value of 255 to get the default blend fraction.

Roads use the night mode color blended with 50% white, and half the specified blend fraction. This increases contrast between roads and
other features.
*/
@property CtColor nightModeColor;

/**
The blend style. To remove the current blend style, set it to null or empty.
If night mode is on, night mode is turned off and replaced by the specified blend style or by no blend style.
*/
@property (nonatomic) NSMutableArray<CtBlendStyle*>* blendStyleSet;

/**
The preferred type of router to be used for calculating routes.
Different routers have various advantages and disadvantages, such as using more or
less memory, being faster or slower, or producing more useful routes.
 
Setting the preferred router type causes navigation data to be unloaded if the current navigation
data uses a different type of router.
*/
@property (nonatomic) CtRouterType preferredRouterType;

/**
The actual router type.

If route data has been loaded for the current map or maps,
returns the actual router type in use, which may not be the same
as the preferred router type, because some router types are not
available for certain types of data.
 
If the route data has not been loaded, returns RouterTypeDefault.
Route data is loaded either when the first route is constructed, or
when LoadNavigationData is called.
*/
@property (nonatomic) CtRouterType actualRouterType;

/**
Starts navigation from a start point to an end point, which may be in different coordinate systems,
and causes a route to be calculated and displayed.
*/
-(CtResult)startNavigationFrom:(CtPoint)aStart startCoordType:(CtCoordType)aStartCoordType to:(CtPoint)aEnd endCoordType:(CtCoordType)aEndCoordType;

/**
Starts navigation through a set of at least two points, taken in order, and causes a route to be calculated and displayed.
*/
-(CtResult)startNavigation:(CtRouteCoordSet*)aCoordSet;

/**
Starts navigation through a set of at least two points, taken in order, and causes a route to be calculated and displayed.
Calculates the initial route asynchronously. It is displayed after the first call to navigate() after it has been calculated.
*/
-(CtResult)startNavigationDeferred:(CtRouteCoordSet*)aCoordSet;

/**
Stops navigating. Current routes remain in existence.
Navigation can be resumed using startNavigation
or enableNavigation.
*/
-(void)endNavigation;

/** Creates a route without starting navigation. */
-(CtRoute*)createRouteWithProfile:(CtRouteProfile*)aProfile coordSet:(CtRouteCoordSet*)aCoordSet;

/**
Creates a route asynchronously and calls aCallback (from a different thread) with the created route and an error code.
 
If aOverride is true, any pending asynchronous routing operations that have not yet been started
are deleted. Asynchronous routing operations are executed in order.
*/
-(CtResult)createRouteAsync:(id<CtCreateRouteAsyncProtocol>)aCallback profile:(CtRouteProfile*)aProfile coordSet:(CtRouteCoordSet*)aCoordSet override:(bool)aOverride;

/**
Creates a route by reading it from XML data in GPX or CartoType route format.
If the parameter aFileNameOrData starts with a less-than sign, it is treated as XML data, otherwise it is treated as a filename.
 
If GPX data contains route elements, the first of them is used; otherwise the first track element is used if any.
 
The profile is ignored if the data is in CartoType route format, which contains the profile used to create it.
*/
-(CtRoute*)createRouteFromXml:(NSString*)aFileNameOrData profile:(CtRouteProfile*)aProfile;

/**
Creates an optimized route to visit a series of waypoints in any order.

If aStartFixed is true, the first point in the route is always the first point supplied.
If aEndFixed is true, the last point in the route is always the last point supplied.
To create a circular route starting and ending at a predetermined place, but visiting the other points
in any order, set aStartFixed and aEndFixed true, and make the first and last point the same.
aIterations is the number of different routes to be tried.

To use the route for navigation, call the useRoute method.
*/
-(CtRoute*)createBestRouteWithProfile:(CtRouteProfile*)aProfile coordSet:(CtRouteCoordSet*)aCoordSet startfixed:(bool)aStartFixed endfixed:(bool)aEndFixed iterations:(int)aIterations;

/**
An asynchronous version of CreateBestRoute.
Creates a route asynchronously and calls aCallback (from a different thread) with the created route and an error code.
 
If aOverride is true, any pending asynchronous routing operations that have not yet been started
are deleted. Asynchronous routing operations are executed in order.
*/
-(CtResult)createBestRouteAsync:(id<CtCreateRouteAsyncProtocol>)aCallback profile:(CtRouteProfile*)aProfile coordSet:(CtRouteCoordSet*)aCoordSet startfixed:(bool)aStartFixed endfixed:(bool)aEndFixed iterations:(int)aIterations override:(bool)aOverride;

/**
Whether navigation is enabled. When navigation is disabled,
the follow modes work but no navigation is performed even if there is a route.
The return value is the previous state.
*/
@property (nonatomic) bool navigationEnabled;

/**
Loads navigation data for the current map. This function need not be called; if it
is not called, navigation data will be loaded automatically when it is first needed.
It is provided so that the time overhead of loading navigation data can be incurred
at a predictable stage, for example at application startup.
 
No error is returned if navigation data has already been loaded.
*/
-(CtResult)loadNavigationData;    

/**
Sets the routing profile. If you do not call this function, the
standard profile for private car routing is used.

This function has no effect if routing is performed on a map
with a built-in contraction-hierarchy routing table (created using makemap /route options ch, cs, ct, tech or tt),
because such a table is created using one or more fixed routing profiles which can be set in makemap.
Use setBuiltInProfile for contraction hierarchy routing.
*/
-(CtResult)setRouteProfile:(CtRouteProfile*)aProfile;

/** Sets the routing profile by type. */
-(CtResult)setRouteProfileType:(CtRouteProfileType)aProfileType;

/**
Returns the current route profile.
*/
-(CtRouteProfile*)routeProfile;

/**
The number of built-in routing profiles owned by the router.
It is zero for routers allowing an arbitrary profile to be set, and one or greater
for the contraction hierarchy router, which stores one or more fixed profiles.
*/
@property (nonatomic) int builtInRouteProfileCount;

/**
Returns a built-in routing profile selected by its index; returns null if the index is out of range.
Routers allowing an arbitrary profile to be set have no built-in profiles.
The contraction hierarchy router has one or more built-in profiles.
*/
-(CtRouteProfile*)builtInRouteProfile:(int)aIndex;

/**
Selects a built-in routing profile by its index; returns an error code if the index is out of range.
Routers allowing an arbitrary profile to be set have no built-in profiles.
The contraction hierarchy router has one or more built-in profiles.
*/
-(CtResult)setBuiltInRouteProfile:(int)aIndex;

/**
Creates a route profile of a given type so that it can be modified as desired and
then used for routing.
*/
+(void)createRouteProfile:(CtRouteProfile*)aProfile type:(CtRouteProfileType)aProfileType;

/** Reads a route profile from a file in XML form. */
+(CtResult)readRouteProfileFromXmlFile:(CtRouteProfile*)aProfile filename:(NSString*)aFileName;

/** Reads a route profile from a string in XML form. */
+(CtResult)readRouteProfileFromXmlString:(CtRouteProfile*)aProfile filename:(NSString*)aString;

/** Writes a route profile to a file in XML form. */
+(CtResult)writeRouteProfileAsXmlFile:(CtRouteProfile*)aProfile filename:(NSString*)aFileName;

/** Writes a route profile to a string in XML form. */
+(NSString*)writeRouteProfileAsXmlString:(CtRouteProfile*)aProfile;

/**
Returns true if navigation is happening.
This state occurs between calls to startNavigation and endNavigation.
*/
@property (nonatomic) bool navigating;

/** If navigating, returns the start of the route in map coordinates. Returns (0,0) if not navigating. */
@property (nonatomic) CtPoint routeStart;

/** If navigating, returns the end of the route in map coordinates. Returns (0,0) if not navigating. */
@property (nonatomic) CtPoint routeEnd;

/**
If aEnable is true, displays the current route and up to two alternative routes.
If aEnable is false, removes the current route from the display.
*/
-(CtResult)displayRoute:(bool)aEnable;

/** Returns the current route, or null if there is none. */
@property (nonatomic) CtRoute* route;

/** Returns the current route segment, or null if not navigating. */
@property (nonatomic) CtRouteSegment* currentRouteSegment;

/**
Reads a route from XML (a GPX file or a CartoType route file) and displays it.
aFileNameOrData is treated as XML data if it starts with a less-than sign.

This function resets the navigation state and enables navigation.
*/
-(CtResult)readRouteFromXml:(NSString*)aFileNameOrData;

/** Returns instructions for a route in the language of the current locale. */
-(NSString*)routeInstructions:(CtRoute*)aRoute;

/**
Loads an existing route and displays it.
 
This function resets the navigation state.
*/
-(CtResult)useRoute:(CtRoute*)aRoute;

/**
Displays a number of route choices, using the "route/choice" layer. Deletes the current route.
This function is intended to allow the user to choose from alternative routes such as fastest and shortest.
Routes need not start and end at the same place, so the function can be used for any purpose where a choice of routes is needed.
*/
-(CtResult)displayRouteChoices:(NSArray<CtRoute*>*)aRouteArray;

/**
Reverses the order of the current route points, recreates the route,
and displays it on the map. The two directions of the same route may
not follow the same path because of one-way streets and turn restrictions.
*/
-(CtResult)reverseRoute;

/** Deletes the route if any. */
-(CtResult)deleteRoute;

/** Writes a route as XML in the format selected by aFileType: Ctroute or Gpx. */
-(CtResult)writeRouteAsXml:(CtRoute*)aRoute file:(NSString*)aFileName fileType:(CtFileType)aFileType;

/** Writes a route as an XML string, in the format selected by aFileType: CtRoute or Gpx. */
-(NSString*)writeRouteAsXmlString:(CtRoute*)aRoute fileType:(CtFileType)aFileType;

/**
Updates the vehicle position and, if navigating, updates the navigation state.

In order to implement turn-by-turn navigation, you should call navigate
at frequent intervals (once per second is good enough) after creating a route
using startNavigation, readRouteFromXml, etc.

When not navigating, you may call this function
to update the vehicle position if necessary (i.e., if the current follow mode
displays the vehicle position: see followMode).

aNavData->validity is a set of bit flags, ORed together, which determines which of
the other parameters are valid. The bit flags are the constants KTimeValid,
KPositionValid, KSpeedValid, KCourseValid, and KHeightValid.
For normal navigation not all the data is necessary:
you need supply the time and position only. Other
data, if available, helps the navigation system decide which road the vehicle
is on, and extrapolate positions. The height (elevation above sea level) is
not currently used.

aNavData->time is the time in seconds since midnight, 1st January 1970.

aNavData->longitude and aNavData->latitude are the location in degrees longitude and latitude, using the WGS84 datum.

aNavData->speed is the speed in kilometers per hour.

aNavData->course is the true course in degrees, where 0 is north, and degrees increase clockwise;
90 is east, 180 is south, and 270 is west.

aNavData->height is the height above sea level in meters.

If the position is invalid, navigate guesses the position by extrapolating from
the current speed, if known, and the previous position.

After navigate has been called, the map position, rotation or zoom will
change according to the current follow mode (see followMode).

If you are not using graphics acceleration you should redraw the map after calling this function.
*/
-(CtResult)navigate:(CtNavigationData*)aNavData;

/**
When navigating, returns the first significant turn after the current position, ignoring 'ahead' and 'continue' turns.
If its type is TurnTypeNone there are no remaining significant turns.
The distance member of the turn is the distance from the current position to the first turn or the arrival point.
*/
@property (nonatomic) CtTurn* firstTurn;

/**
When navigating, returns any significant turn 100 meters or less after the first turrn.
If its type is TurnTypeNone there is no such turn.
The distance member of the turn is the distance from the first turn to the second turn.
*/
@property (nonatomic) CtTurn* secondTurn;

/**
When navigating, returns any 'ahead' or 'continue' turn before the first turn, so that the user interface can
show that no turn is required at the next junction or adjoining minor road.
If its type is TurnTypeNone there is no such turn.
The distance member of the turn is the distance from the current position to the continuation turn.
*/
@property (nonatomic) CtTurn* continuationTurn;

/** Returns the current navigation state. */
@property (nonatomic) CtNavigationState navigationState;

/** When navigating, the voice instruction to be issued, if any, after a call to Navigate. */
@property (nonatomic) NSString* voiceInstruction;

/** The minimum distance in meters between location fixes in meters that is taken as an actual move when navigating. */
@property (nonatomic) int navigationMinimumFixDistance;

/** The maximum time off route in seconds before a new route is calculated when navigating. */
@property (nonatomic) int navigationTimeOffRouteTolerance;

/** The maximum distance from the route in meters before the vehicle is deemed off-route when navigating. */
@property (nonatomic) int navigationDistanceOffRouteTolerance;

/** Sets parameters affecting how navigation works. */
@property (nonatomic) CtNavigatorParam* navigatorParam;

/**
The current location match parameters.
Setting them to null causes default parameters are used.
These parameters are used when creating routes in functions which do not receive explicit location match parameters.
In functions which do receive location match parameters, any parameters with the value 0 are replaced by
the current location match parameters set by this function.
*/
@property (nonatomic) CtLocationMatchParam* locationMatchParam;

/**
Adds or replaces a nearby object warning.
Objects found are added to the 'nearby' layer while the warning is in force.
The attribute '_warning=0' is set.

Objects in aLayer, satisfying aCondition if it is not empty, and within aMaxDistance of the current point, are highlighted.
The parameter aLayer may contain wild cards or end in a '/' sign.

The parameter aMaxObjectCount is the maximum number of objects to be found, which is clamped to the range
0...1024. If it is zero, the default of 32 is used. If more objects are actually found, the closest ones are used.

If aId is zero, a new warning is created. If aId is non-zero, the existing warning if any with an ID of aId is replaced.
The ID of the new or replaced warning is returned. If aMaxDistance, which is in meters, is less than or equal to zero,
it is set to the default distance of 1000, or 1km.

Returns 0 if no warning was created, which occurs if aLayer was empty, or the navigation system could not be created.
*/
-(uint32_t)setNearbyObjectWarning:(uint32_t)aId layer:(NSString*)aLayer condition:(NSString*)aCondition maxDistance:(double)aMaxDistance maxObjectCount:(int)aMaxObjectCount;

/**
Adds a vehicle type warning and returns its ID.
Road sections illegal for the current vehicle weight, dimensions or cargo type are found.

Objects found are added to the 'nearby' layer while the warning is in force.
The attribute '_warning=1' is set.

The parameter aMaxObjectCount is the maximum number of objects to be found, which is clamped to the range
0...1024. If it is zero, the default of 32 is used. If more objects are actually found, the closest ones are used.

If there is already a vehicle type warning it is replaced by this one.

Returns 0 if no warning was created, which occurs if aLayer was empty, or the navigation system could not be created.
*/
-(uint32_t)setVehicleTypeWarning:(double)aMaxDistance maxObjectCount:(int)aMaxObjectCount;

/** Deletes the nearby object warning or vehicle type warning with an ID of aId and returns true if any such object existed. */
-(bool)deleteNearbyObjectWarning:(uint32_t)aId;

/** Clears all nearby object warnings, and the objects themselves, and returns true if any objects were deleted. */
-(bool)clearNearbyObjectWarnings;

/** Returns copies of all the objects for which nearby object warnings exist. */
-(NSMutableArray<CtMapObject*>*)copyNearbyObjects;

/** When navigating, returns the current distance to the destination in meters. */
@property (nonatomic) double distanceToDestination;

/** When navigating, returns the estimated time to the destination in seconds. */
@property (nonatomic) double estimatedTimeToDestination;

/**
Finds the nearest road to the point aPoint, returns information about it in aInfo,
and optionally (if aDisplayPosition is true) sets the current location to that position.
Uses the current location match parameters.
 
Roads are defined as map objects allowed by the current routing filter and main route profile.
If aHeadingInDegrees is valid (not negative) it is used to choose among roads going in different directions.
*/
-(CtResult)findNearestRoad:(CtNearestRoadInfo*)aInfo point:(CtPoint)aPoint coordType:(CtCoordType)aCoordType headingInDegrees:(double)aHeadingInDegrees displayPosition:(bool)aDisplayPosition;

/**
Moves the route position object and the route vector object to the nearest point on a road.

Uses the supplied navigation data and the current location match parameters. If the course (heading) is supplied
it is used to choose among roads going in different directions.
If aInfo is non-null, it is used to return information about the nearest road.
This function can be used whether or not navigation is being performed.
*/
-(CtResult)displayPositionOnNearestRoad:(CtNavigationData*)aNavData nearestRoadInfo:(CtNearestRoadInfo*)aInfo;

/**
Sets the vehicle position when navigating as an offset from the center of the display,
in units of the width and height of the display. For example, to show the vehicle
in the middle but three quarters of the way down the display, use (0,0.25).
*/
-(void)setVehiclePosOffsetX:(double)aXOffset andY:(double)aYOffset;

/** The follow mode, which controls the way the map tracks the user location and heading. */
@property (nonatomic) CtFollowMode followMode;

/**
Sets the scale denominators to be used during navigation when moving at various speeds.
aMinScale is the minimum scale; aSlowScale is used at 20kph; aFastScale is used at 80kph; aMaxScale is the maximum scale.
*/
-(CtResult)setFollowScale:(double)aMinScale slowScale:(double)aSlowScale fastScale:(double)aFastScale maxScale:(double)aMaxScale;

/** Gets the current position used by the navigation system. This may be an extrapolated position. */
-(CtResult)getNavigationPosition:(CtPoint*)aPoint coordType:(CtCoordType)aCoordType;

/**
The locale used for map object names and route instructions.
A locale is usually a two-letter language code like "en" (English) or "fr" (French).
If the locale is null or empty, standard names are used for map objects, and English for route instructions.
*/
@property (nonatomic) NSString* locale;

/** Whether distance units are metric (kilometers) or non-metric (miles). */
@property (nonatomic) bool metricUnits;

/** Returns a distance as a string containing locale-dependent words for kilometres, metres, miles and yards. */
-(NSString*)distanceToString:(double)aDistanceInMeters abbreviate:(bool)aAbbreviate;

/** Returns a time duration as a string containing locale-dependent words for hours, minutes and seconds. */
-(NSString*)timeToString:(double)aTimeInSeconds;

/**
Sets the letter case of a string.

Supports all UTF-16 characters. Uses the title-case exception dictionary to prevent words like 'of' being capitalised.
Returns the modified string. Does not change aString.
*/
-(NSString*)setCase:(NSString*)aString case:(CtLetterCase)aCase;

/**
Adds traffic information, such as a speed restriction, prohibition on the use of a route, reduction in
the number of lanes, notification of road works, etc. At present only speed restrictions are supported.
 
This is the general function.
See also the specialized functions addPolygonSpeedLimit, addLineSpeedLimit, addClosedLineSpeedLimit and addForbiddenArea.

The objectID field of the return value is used to return an identifier for the information, which can be used when calling deleteTrafficInfo().
If aId is non-zero when passed in, it is used, and any existing object with that ID is replaced, otherwise a new identifier is generated.

aTrafficInfo gives the type of information (speed limit, etc.).

aLocationRef defines the location (area, road, point, etc.). Only the values ERoadOrientationForwards and ERoadOrientationNone
(which means the same in this context) is supported for iRoadOrientation in aLocationRef.
*/
-(CtIdAndResult)addTrafficInfo:(uint64_t)aId trafficInfo:(CtTrafficInfo*)aTrafficInfo locationRef:(CtLocationRef*)aLocationRef;

/**
Adds a speed limit in kph, to be used when calculating routes, to all roads in a certain polygon.
To specify particular roads, or an area enclosed by roads, use addTrafficInfo.
This allows the router to take account of slow-downs caused by traffic density, road works, etc.
 
The objectID field of the return value is used to return an identifier for the information, which can be used when calling deleteTrafficInfo().
If aId is non-zero when passed in, it is used, and any existing object with that ID is replaced, otherwise a new identifier is generated.
 
aSpeed is the speed to which travel is restricted, in kph. If aSpeed is zero or less, routing is forbidden in
the given polygon.
 
aVehicleTypes is a set of flags drawn from the CtRouteAccess... constants, as used in CtRouteProfile.
If aVehicleTypes is 0, all vehicles are affected.
 
A polygon map object representing the speed limit is added to the layer 'traffic'.
 
See also addTrafficInfo, addLineSpeedlimit, addForbiddenArea, deleteTrafficInfo, and clearTrafficInfo.
*/
-(CtIdAndResult)addPolygonSpeedLimit:(uint64_t)aId polygon:(CtGeometry*)aPolygon speed:(double)aSpeed vehicleTypes:(uint32_t)aVehicleTypes;

/**
Adds a speed limit in kph, to be used when calculating routes, to a route defined by a series of points.

The objectID field of the return value is used to return an identifier for the information, which can be used when calling deleteTrafficInfo().
If aId is non-zero when passed in, it is used, and any existing object with that ID is replaced, otherwise a new identifier is generated.

aLine is a series of points defining the road. The road is calculated as the best route through the points, using the route network,
so to be certain of specifying the desired route you need to supply a point for every junction. The points are snapped to
the nearest nodes in the route network. This means that at the moment you cannot set a speed limit for a range smaller
than an arc between two nodes.

aSpeed is the speed to which travel is restricted in kph.
Values of zero or less indicate that the road is closed.
Values higher than 255 indicate that there is no speed limit.

aVehicleTypes is a set of flags drawn from the CtRouteAccess... constants, as used in CtRouteProfile.
If aVehicleTypes is 0, all vehicles are affected.

A line map object representing the speed limit is added to the layer 'traffic'.

See also addTrafficInfo, addClosedLineSpeedLimit, addPolygonSpeedlimit, addForbiddenArea, deleteTrafficInfo, and clearTrafficInfo.
*/
-(CtIdAndResult)addLineSpeedLimit:(uint64_t)aId line:(CtGeometry*)aLine speed:(double)aSpeed vehicleTypes:(uint32_t)aVehicleTypes;

/**
Adds a speed limit in kph, to be used when calculating routes, to a polygon which is the interior of a
closed line along roads, defined by a series of points.

The objectID field of the return value is used to return an identifier for the information, which can be used when calling deleteTrafficInfo().
If aId is non-zero when passed in, it is used, and any existing object with that ID is replaced, otherwise a new identifier is generated.

aLine is a series of points defining the polygon. The polygon is calculated as the best route through the points, using the route network,
so to be certain of specifying the desired polygon you need to supply a point for every junction.

aSpeed is the speed to which travel is restricted in kph.
Values of zero or less indicate that the road is closed.
Values higher than 255 indicate that there is no speed limit.

aVehicleTypes is a set of flags drawn from the KRouteAccess... constants, as used in CtRouteProfile.
If aVehicleTypes is 0, all vehicles are affected.

A polygon map object representing the speed limit is added to the layer 'traffic'.

See also addTrafficInfo, addLineSpeedLimit, addPolygonSpeedlimit, addForbiddenArea, deleteSpeedLimit, and clearTrafficInfo.
*/
-(CtIdAndResult)addClosedLineSpeedLimit:(uint64_t)aId line:(CtGeometry*)aLine speed:(double)aSpeed vehicleTypes:(uint32_t)aVehicleTypes;

/**
Marks a certain polygon as forbidden for routing.

The objectID field of the return value is used to return an identifier for the information, which can be used when calling deleteTrafficInfo().
If aId is non-zero when passed in, it is used, and any existing object with that ID is replaced, otherwise a new identifier is generated.

See also addTrafficInfo, addLineSpeedLimit, addPolygonSpeedLimit, deleteTrafficInfo and clearTrafficInfo.
*/
-(CtIdAndResult)addForbiddenArea:(uint64_t)aId polygon:(CtGeometry*)aPolygon;

/**
Deletes traffic information: a speed limit, forbidden area, or other information, referring
to it by the ID returned when the restriction was added by AddTrafficInfo(), AddLineSpeedLimit(),
AddForbiddenArea(), etc.
 
Use ClearTrafficInfo() to delete all restrictions and forbidden areas.
*/
-(CtResult)deleteTrafficInfo:(uint64_t)aId;

/** Deletes all speed restrictions, forbidden areas and other traffic information. */
-(void)clearTrafficInfo;

/**
Enables or disables the use of traffic information when routing.
*/
@property (nonatomic) bool trafficInfoEnabled;

/**
Starts tracking (storing and displaying track points) and starts a new track segment.
If tracking is turned on, a track point is added every time a new position fix is
supplied using Navigate, if the position is far enough from the previous track point.
*/
-(void)startTracking;

/** Stops tracking (storing and displaying track points). Does not delete the current track. */
-(void)endTracking;

/** Deletes the current track. Does not affect whether tracking is on or off. */
-(void)deleteTrack;

/** Returns true if tracking (storing and displaying track points) is on. */
-(bool)tracking;

/** Enables or disables displaying the track. */
@property (nonatomic) bool displayTrack;

/**
Returns the current track as a geometry object with coordinates in degrees.
Each segment of the track is a contour in the geometry object.
*/
@property (nonatomic) CtGeometry* track;

/** Returns the length of the current track in meters. */
@property (nonatomic) double trackLengthInMeters;

/** Writes the current track to a file as XML in GPX format. */
-(CtResult)writeTrackAsXml:(NSString*)aFileName;

/** Writes the current track to a string as XML in GPX format. */
@property (nonatomic) NSString* trackAsXmlString;

/**
Returns the area and length of a geometry object on the surface of the earth in square meters and meters respectively, using the WGS84 ellipsoid.
The area and length of a point object are both zero.
The area of a line object is zero.
This function ignores curves. All points are treated as on-curve.
*/
-(CtAreaLengthAndResult)areaAndLength:(CtGeometry*)aGeometry;

/**
Returns the area and length of a contour of a geometry object on the surface of the earth in square meters and meters respectively, using the WGS84 ellipsoid.
The area and length of a point are both zero.
The area of a line is zero.
This function ignores curves. All points are treated as on-curve.
*/
-(CtAreaLengthAndResult)contourAreaAndLength:(CtGeometry*)aGeometry contourIndex:(int)aContourIndex;

/**
Returns the azimuth (direction along a great circle line) from one point to another in degrees,
given two lat-long points in degrees.

This function uses the WGS84 ellipsoid.

North is 0 and angles increase clockwise.
*/
+(double)azimuthInDegreesFrom:(CtPoint)aStart to:(CtPoint)aEnd;

/** Finds the point in degrees at a certain distance and azimuth direction from a starting point. This function uses the WGS84 ellipsoid. */
+(CtPoint)pointAtAzimuthFrom:(CtPoint)aPoint dir:(double)aDir distance:(double)aDistanceInMeters;

/**
Finds the great-circle distance in meters between two lat-long points in degrees,
using the WGS84 ellipsoid.
*/
+(double)greatCircleDistanceInMetersFrom:(CtPoint)aStart to:(CtPoint)aEnd;

/** Sends an HTTP request to the specified URL and returns the resulting data. */
+(NSData*)readUrl:(NSString*)aUrl;

/**
Treating a string as a set of map object attributes of the form
label|key1=value1|key2=value2| ... |keyN=valueN, where | is character code 0,
sets a named attribute. An empty or zero-length name sets the label.
An empty or zero-length value deletes a name-value pair.

This function allows you to create a string containing multiple attributes when
adding a new map object at run-time.

Any of aString, aKey and aValue may be null.

A new string with the new key-value pair is returned.
*/
+(NSString*)setAttributeInString:(NSString*)aString withKey:(NSString*)aKey andValue:(NSString*)aValue;

/** Constructs a CartoType color, represented as a 32-bit integer holding 8 bits each for the red, green, blue and alpha values. */
+(CtColor)makeColorR:(int)aR G:(int)aG B:(int)aB A:(int)aA;

/** Constructs a CartoType color value, which is a 32-bit integer, from a string; see the CartoType style sheet documentation for details of the format. */
+(CtColor)makeColor:(NSString*)aColor;

/** Sets the alpha value of a color and returns the new color value. */
+(CtColor)setColor:(CtColor)aColor A:(int)aAlpha;

/** Returns a string description for an error code. */
+(NSString*)errorString:(uint32_t)aCode;

/**
Reads textual data from a URL via HTTP, returning it as a string. For internal use only.
Uses the GET method unless aPostData is non-empty, in which case POST is used.
*/
+(NSString*)readFromUrl:(NSString*)aUrl postData:(NSString*)aPostData;

/**
Converts a point in degrees latitude and longitude (WGS84) into a
UK Ordnance Survey national grid reference, using aDigits as the number of
digits per coordinate; this value is clamped to the range 0...5.
Five-digit references are accurate to a meter.

Returns nil if the point is outside the UK national grid bounds.
*/
+(NSString*)UKGridReferenceFromDegrees:(CtPoint)aPointInDegrees digits:(int)aDigits;

/**
Converts a UK Ordnance Survey national grid reference (two capital letters followed by an even number of digits)
into a point in degrees latitude and longitude (WGS84).

The point returned is the center of the square defined by the grid reference.
Returns the point (0,0) if the grid reference is invalid.
*/
+(CtPoint)PointInDegreesFromUKGridReference:(NSString*)aGridReference;

@end
/** A synonym of CtFramework. */
@compatibility_alias CartoTypeFramework CtFramework;

#ifndef CARTOTYPE_WATCHOS

/** A map renderer using OpenGL ES graphics acceleration. */
@interface CtMapRenderer: NSObject

/**
Constructs a renderer object that can be used to draw a CtFramework into a GLKView (a view which supports OpenGL ES drawing).
The CtMapRenderer draw method should be called by the GLKView drawRect method.
You can call any function on the framework object; functions which affect the view will automatically cause the GLKview to be updated
when drawRect is called.
*/
-(id)init:(CtFramework*)aFramework;

/** Draws the map using OpenGL ES. */
-(void)draw;

@end
/** A synonym of CtMapRenderer. */
@compatibility_alias CartoTypeMapRenderer CtMapRenderer;

#endif
