/*
CartoTypeRouter.h
Copyright 2013-2024 CartoType Ltd.
See www.cartotype.com for more information.
*/

#ifndef CartoTypeRouter_h__
#define CartoTypeRouter_h__

#import <CartoType/CartoTypeBase.h>

/**
\file
Constants, types and interfaces used by routing and navigation.
*/

/** The maximum number of different routes which can be calculated using route profiles and displayed simultaneously. */
static const uint32_t CtMaxRoutesDisplayed = 16;

/** Constants used to select frequently-used route profiles. */
typedef NS_ENUM(NSInteger,CtRouteProfileType)
    {
    /** A profile type for private car navigation. */
    CtRouteProfileTypeCar,
    /** A profile type for walking. */
    CtRouteProfileTypeWalk,
    /** A profile type for cycling. */
    CtRouteProfileTypeBike,
    /** A profile type for cycling. */
    CtRouteProfileTypeCycle = CtRouteProfileTypeBike,
    /** A profile type for walking, preferring off-road paths. */
    CtRouteProfileTypeHike,
    /** <summary> A profile type for downhill skiing. </summary> */
    CtRouteProfileTypeSki
    };
/** A synonym of CtRouteProfileType. */
typedef CtRouteProfileType CartoTypeRouteProfileType;

/**
A route profile: parameters determining the type of route,
including road speeds, bonuses and penalties, and vehicle type.
*/
@interface CtRouteProfile: NSObject
    /** Creates a standard route profile for a car. */
    -(id)init;

    /** Creates a route profile with one of the standard types. */
    -(id)initWithProfileType:(CtRouteProfileType)aProfileType;

    /** The optional name of the profile. */
    @property (nonatomic, strong) NSString* name;
        
    /**
    Flags taken from the KRouteAccess... constants indicating the vehicle type.
    Arcs with restrictions matching any of these flags will not be taken.
    */
    @property (nonatomic) uint32_t vehicleType;
    
    /** The vehicle's weight in metric tons. Values of zero or less mean 'unknown'. */
    @property (nonatomic) double weight;
    
    /** The vehicle's axle load in metric tons. Values of zero or less mean 'unknown'. */
    @property (nonatomic) double axleLoad;
    
    /** The vehicle's double axle load in metric tons. Values of zero or less mean 'unknown'. */
    @property (nonatomic) double doubleAxleLoad;

    /** The vehicle's triple axle load in metric tons. Values of zero or less mean 'unknown'. */
    @property (nonatomic) double tripleAxleLoad;

    /** The vehicle's height in meters. Values of zero or less mean 'unknown'. */
    @property (nonatomic) double height;
    
    /** The vehicle's width in meters. Values of zero or less mean 'unknown'. */
    @property (nonatomic) double width;
    
    /** The vehicle's length in meters. Values of zero or less mean 'unknown'. */
    @property (nonatomic) double length;
    
    /** True if the vehicle is carrying hazardous materials. */
    @property (nonatomic) bool hazMat;
    
    /** Returns the expected speed for a road type in kilometres per hour.  The road type aIndex must be in the the range FeatureTypeMotorway (0) to FeatureTypeUnknownRoute (31). */
    -(double)speed:(unsigned int)aIndex;
    /** Sets the expected speed for a road type in kilometres per hour.   The road type aIndex must be in the the range FeatureTypeMotorway (0) to FeatureTypeUnknownRoute (31).*/
    -(void)setSpeed:(unsigned int)aIndex to:(double)aValue;
    /**
    Returns the bonus or penalty in notional km per hour to be added to a road type to make it more or less likely to be used.
    The road type aIndex must be in the the range FeatureTypeMotorway (0) to FeatureTypeUnknownRoute (31).
    For example, a penalty of 1kph is applied to walking routes along major roads because it is pleasanter to walk along quieter minor roads.
    */
    -(double)bonus:(unsigned int)aIndex;
    /** Sets the bonus for a road type. The road type aIndex must be in the the range  The road type aIndex must be in the the range FeatureTypeMotorway (0) to FeatureTypeUnknownRoute (31).. */
    -(void)setBonus:(unsigned int)aIndex to:(double)aValue;

    /**
    Returns the restriction override bit mask for a road type. The road type aIndex must be in the the range FeatureTypeMotorway (0) to FeatureTypeUnknownRoute (31).
    */
    -(uint32_t)restrictionOverride:(unsigned int)aIndex;
    /**
    Sets the restriction override bit mask for a road type. The road type aIndex must be in the the range FeatureTypeMotorway (0) to FeatureTypeUnknownRoute (31).
    The bit mask allows restrictions to be overridden for certain
    types of road. For example, to allow routing of vehicles along
    tracks, even if they are forbidden to motor vehicles, call setRestrictionOverride(FeatureTypeTrack,KRouteAccessMotorVehicleFlag).
    */
    -(void)setRestrictionOverride:(unsigned int)aIndex to:(uint32_t)aValue;

    /**
    The estimated time in seconds taken for any turn at a junction that is not
    a slight turn or simply going straight ahead.
    */
    @property (nonatomic) int turnTime;
    
    /**
    The estimated time in seconds taken for a U-turn, defined as a turn
    very close to 180 degrees (within 1/32 of a circle = 11.75 degrees).
    */
    @property (nonatomic) int uTurnTime;

    /**
    The estimated time in seconds taken for a turn across the traffic:
    that is, a left turn in drive-on-right countries,
    or a right turn in drive-on-left countries. The time is used when turning
    from the right of way to a lower-rank road. It is doubled when
    turning from a lower-rank road on to the right of way.
    */
    @property (nonatomic) int crossTrafficTurnTime;
    
    /**
    The estimated delay in seconds caused by traffic lights.
    */
    @property (nonatomic) int trafficLightTime;
    
    /** Set this flag to true to get the shortest route by distance or time, ignoring weightings. Whether distance or time is used depends on the setting of shortestByTime. */
    @property (nonatomic) bool shortest;
    
    /** Set this flag to true if iShortest should produce the shortest route by time rather than distance. */
    @property (nonatomic) bool shortestByTime;

    /**
    The penalty applied to toll roads as a number between zero and one.
    The value 1 means that no toll roads will be used, 0.5 makes toll roads half as desirable as non-toll roads
    of the same road type, and so on. The value 0 means that no penalty is applied to toll roads.
    Values outside the range 0...1 are clamped to that range.
    */
    @property (nonatomic) double tollPenalty;

    /**
    Returns the speed added to a road with a particular gradient. The gradient type aIndex must be in the range 0...7.
    These values are intended for cycling, walking, hiking and running profiles,
    where speeds are generally the same for different types of road, but
    can change according to the gradient.
    */
    -(double)gradientSpeed:(unsigned int)aIndex;
    /** Sets the speed added to a road with a particular gradient. The gradient type aIndex must be in the range 0...7. */
    -(void)setGradientSpeed:(unsigned int)aIndex to:(double)aValue;

    /** Returns the bonus added to a road with a particular gradient. The gradient type aIndex must be in the range 0...7. */
    -(double)gradientBonus:(unsigned int)aIndex;
    /** Sets the bonus added to a road with a particular gradient. The gradient type aIndex must be in the range 0...7. */
    -(void)setGradientBonus:(unsigned int)aIndex to:(double)aValue;
    
    /**
    Flags indicating which roads are affected by gradient speeds and bonuses; normally steps, etc., are excluded.
    Each bit is (1 << road type).
    */
    @property (nonatomic) uint32_t gradientFlags;

    /** (for internal use) Returns a pointer to the underlying C++ profile object. */
    -(void*)profile;
@end
/** A synonym of CtRouteProfile. */
@compatibility_alias CartoTypeRouteProfile CtRouteProfile;

/** Types of router; used when selecting a router type. */
typedef NS_ENUM(NSInteger,CtRouterType)
    {
    /** This router type causes the default router to be selected: the one for which serialized data is available in the map file, or, failing that, RouterTypeStandardAStar. */
    CtRouterTypeDefault,
    /** The A* router, which gives fast performance but takes a lot of memory and cannot create a route going twice through the same junction. */
    CtRouterTypeStandardAStar,
    /** A version of the A* router with road nodes and turn arcs; slower than RouterTypeStandardAStar, and uses more memory, but can generate routes going twice through the same junction, for more flexible routing. */
    CtRouterTypeTurnExpandedAStar,
    /**
    The contraction hierarchy router is intended where less RAM is available: for example with large maps on mobile devices.
    It gives the same routes as RouterTypeStandardAStar, but is a little slower and does not support custom route profiles; the route profile is decided at the time of creating the CTM1 file.
    */
    CtRouterTypeStandardContractionHierarchy,
    /** Turn-expanded contraction hierarchy. */
    CtRouterTypeTECH
    };
/** A synonym of CtRouterType. */
typedef CtRouterType CartoTypeRouterType;

/**
Turn types on a route. Turns at junctions are classified
by dividing the full circle into 45-degree segments,
except for sharp turns left and right, which encompass everything
up to an actual U-turn, and left and right forks, for which there
are special rules to avoid classifying them as ahead.
 
The actual turn angle is also given in the CtTurn class.
*/
typedef NS_ENUM(NSInteger,CtTurnType)
    {
    /** No turn exists or is needed. This turn type is used at the start of a route. */
    CtTurnTypeNone,
    /**
    A turn of less than 22.5 degrees left or right, unless this is a fork with two choices, in which
    case the turn will be bear right or bear left.
    */
    CtTurnTypeAhead,
    /**
    A turn between 22.5 degrees and 67.5 degrees right
    or a turn through a smaller angle which is the rightmost of a fork with two choices.
    */
    CtTurnTypeBearRight,
    /** A turn between 67.5 degrees and 112.5 degrees right. */
    CtTurnTypeRight,
    /** A turn between 112.5 degrees and 180 degrees right. */
    CtTurnTypeSharpRight,
    /** This turn type is use for U-turns: turns back along the same road. */
    CtTurnTypeAround,
    /** A turn between 112.5 degrees and 180 degrees left. */
    CtTurnTypeSharpLeft,
    /** A turn between 67.5 degrees and 112.5 degrees left. */
    CtTurnTypeLeft,
    /**
    A turn between 22.5 degrees and 67.5 degrees left.
    or a turn through a smaller angle which is the leftmost of a fork with two choices.
    */
    CtTurnTypeBearLeft
    };
/** A synonym of CtTurnType. */
typedef CtTurnType CartoTypeTurnType;

/**
Turns involving roundabouts are marked as such so that
exit numbers can be counted
*/
typedef NS_ENUM(NSInteger,CtRoundaboutState)
    {
    /** This junction does not involve a roundabout. */
    CtRoundaboutStateNone,
    /** This junction enters a roundabout. */
    CtRoundaboutStateEnter,
    /** This junction continues around a roundabout. */
    CtRoundaboutStateContinue,
    /** This junction exits a roundabout. */
    CtRoundaboutStateExit
    };
/** A synonym of CtRoundaboutState. */
typedef CtRoundaboutState CartoTypeRoundaboutState;

/** Information about a turn from one route segment to another. */
@interface CtTurn: NSObject
    /** The turn type: ahead, left, right, etc. */
    @property (nonatomic) CtTurnType turnType;
    /** True if this turn is a continuation of the current road and no notification is needed. */
    @property (nonatomic) bool isContinue;
    /** The roundabout state. */
    @property (nonatomic) CtRoundaboutState roundaboutState;
    /** The turn angle: 0 = straight ahead; negative = left, positive = right. */
    @property (nonatomic) double turnAngle;
    /** The junction to take, counting the current junction as 0, if this junction is part of a roundabout. */
    @property (nonatomic) int exitNumber;
    /** The distance in meters from the current position to the turn or the arrival point. */
    @property (nonatomic) double distance;
    /** The estimated time in seconds from the current position to the turn or the arrival point. */
    @property (nonatomic) double time;
    /** The name of the road before the turn. */
    @property (nonatomic, strong) NSString* fromName;
    /** The reference code (e.g., B4009) of the road before the turn. */
    @property (nonatomic, strong) NSString* fromRef;
    /** The road type of the road before the turn. */
    @property (nonatomic) CtFeatureInfo* fromRouteInfo;
    /** The name of the road after the turn. */
    @property (nonatomic, strong) NSString* toName;
    /** The reference code (e.g., B4009) of the road after the turn. */
    @property (nonatomic, strong) NSString* toRef;
    /** The road type of the road after the turn. */
    @property (nonatomic) CtFeatureInfo* toRouteInfo;
    /** The destination of the road after the turn. */
    @property (nonatomic, strong) NSString* destination;
    /** The position of the turn in map coordinates. */
    @property (nonatomic) CtPoint position;
    /** Instructions for the turn in the language of the current locale. */
    @property (nonatomic, strong) NSString* instructions;
    /** The index of the turn, which is unique within a route. Used to avoid duplication when issuing voice instructions. */
    @property (nonatomic) int index;
@end
/** A synonym of CtTurn. */
@compatibility_alias CartoTypeTurn CtTurn;

/** The side of the road: used in traffic information. */
typedef NS_ENUM(NSInteger,CtSideOfRoad)
    {
    CtSideOfRoadNone,
    CtSideOfRoadRight,
    CtSideOfRoadLeft,
    CtSideOfRoadBoth
    };
/** A synonym of CtSideOfRoad. */
typedef CtSideOfRoad CartoTypeSideOfRoad;

/** Information about the nearest road to a point. */
@interface CtNearestRoadInfo: NSObject
    /** The road type of the arc. */
    @property (nonatomic, strong) CtFeatureInfo* featureInfo;
    /**
    The maximum legal speed in kilometres per hour. A value of zero means there is no known speed limit,
    or the standard speed limit for the type of road applies.
    */
    @property (nonatomic) double maxSpeed;
    /** The standard name of the road. */
    @property (nonatomic, strong) NSString* name;
    /** The road reference of the road. */
    @property (nonatomic, strong) NSString* ref;
    /** The nearest point on the road, in map coordinates, to the chosen point. */
    @property (nonatomic) CtPoint nearestPoint;
    /** The distance from the chosen point to the nearest point in meters. */
    @property (nonatomic) double distance;
    /** The heading of the nearest segment in degrees. */
    @property (nonatomic) double headingInDegrees;
    /** The heading of the nearest segment as a unit vector. */
    @property (nonatomic) CtPoint headingVector;
    /**
    The side of the road on which the selected point lies, or SideOfRoadNone if the vehicle heading is not supplied.
    It is Left if the heading vector is left of a vector from the chosen point to the nearest point on the road, otherwise Right.
    Another way to understand it is: if the road as seen by the driver of the vehicle is to the left, the side of the road is Right, and vice versa.
    */
    @property (nonatomic) CtSideOfRoad sideOfRoad;
@end
/** A synonym of CtNearestRoadInfo. */
@compatibility_alias CartoTypeNearestRoadInfo CtNearestRoadInfo;

/** Parameters used when matching a road or other feature to a location. */
@interface CtLocationMatchParam: NSObject
    /** Creates a CtLocationMatchParam object with default values. */
    -(id)init;
    /**
    The accuracy of a location fix given as a range error with 95% probability.
    It is clamped to the range 1 ... 1000.
    A value of zero indicates the default value: 8.
    */
    @property (nonatomic) double locationAccuracyInMeters;
    /**
    The accuracy of a heading or course given as an angular error in degrees with 95% probability.
    It is clamped to the range 1 ... 90.
    A value of zero indicates the default value: 22.5.
    */
    @property (nonatomic) double headingAccuracyInDegrees;
    /**
    The expected maximum distance of a road from the current location.
    It is clamped to the range 5 ... 10,000.
    A value of zero indicates the default value: 100.
    */
    @property (nonatomic) double maxRoadDistanceInMeters;
@end
/** A synonym of CtLocationMatchParam. */
@compatibility_alias CartoTypeLocationMatchParam CtLocationMatchParam;

/** A point on a route, with its heading and location match parameters. */
@interface CtRoutePoint: NSObject
    /** The point. */
    @property (nonatomic) CtPoint point;
    /** The heading in degrees clockwise from north. */
    @property (nonatomic) double heading;
    /** True if the heading is known. */
    @property (nonatomic) bool headingKnown;
    /** Parameters used when matching the point to a road or other routable segment. If it is set to null, default parameters are used. */
    @property (nonatomic, strong) CtLocationMatchParam* locationMatchParam;
    /** For internal use: returns a pointer to the RoutePoint object. */
    -(void*)routePoint;
@end
/** A synonym of CtRoutePoint. */
@compatibility_alias CartoTypeRoutePoint CtRoutePoint;

/**
An ordered set of points for creating a route, with optional heading and accuracy information.
Headings are used where available, and where possible, to decide between
roads or choose the direction of travel on a road.
*/
@interface CtRouteCoordSet: NSObject
    /** The coordinate type of the route points. */
    @property (nonatomic) CtCoordType coordType;
    /** Appends a route point. */
    -(void)append:(CtRoutePoint*)aPoint;
    /** Returns the number of route points. */
    -(int)pointCount;
    /** Returns the route point at a given index. */
    -(CtRoutePoint*)point:(int)aIndex;
    /** For internal use: returns a pointer to the TRouteCoordSet object. */
    -(void*)routeCoordSet;
@end
/** A synonym of CtRouteCoordSet. */
@compatibility_alias CartoTypeRouteCoordSet CtRouteCoordSet;

/** An ordered set of points. */
@interface CtPointSet: NSObject
    /** Creates a CtPointSet with a given coordinate type. */
    -(id)init:(CtCoordType)aCoordType;
    /** The coordinate type of the points. */
    @property (nonatomic, readonly) CtCoordType coordType;
    /** Appends a point. */
    -(void)append:(CtPoint)aPoint;
    /** Returns the number of points. */
    -(int)pointCount;
    /** Returns the route point at a given index. */
    -(CtPoint)point:(int)aIndex;
    /** For internal use: returns a pointer to the point vector. */
    -(void*)pointVector;
@end
/** A synonym of CtPointSet. */
@compatibility_alias CartoTypePointSet CtPointSet;

/** Flags used in CtNavigationData.validity. */
typedef NS_OPTIONS(uint32_t,CtNavigationDataValid)
    {
    /** The time is valid. */
    CtNavigationDataValidTime = 1,
    /** The position is valid. */
    CtNavigationDataValidPosition = 2,
    /** The speed is valid. */
    CtNavigationDataValidSpeed = 4,
    /** The course is valid. */
    CtNavigationDataValidCourse = 8,
    /** The height is valid. */
    CtNavigationDataValidHeight = 16
    };
/** A synonym of CtNavigationDataValid. */
typedef CtNavigationDataValid CartoTypeNavigationDataValid;

/** Data representing a location fix, for example information received from a GPS device. */
typedef struct
    {
    /** Flags indicating validity or availability of data. */
    CtNavigationDataValid validity;
    /**
    The time in seconds. Any consistent time representation may be used (e.g., the unix
    timestamp format of the time in seconds since midnight, 1st Jan 1970); the absolute
    time does not matter as long as times can be subtracted to give a difference in seconds.
    */
    double time;
    /** Position degrees longitude. */
    double longitude;
    /** Position degrees latitude. */
    double latitude;
    /** Speed in kilometres per hour. */
    double speed;
    /** True course in degrees. */
    double course;
    /** Height above sea level in metres. */
    double height;
    } CtNavigationData;
/** A synonym of CtNavigationData. */
typedef CtNavigationData CartoTypeNavigationData;

/** States of the navigation system. */
typedef NS_ENUM(NSInteger,CtNavigationState)
    {
    /** No route has been created, or navigation is disabled. */
    CtNavigationStateNone,
    /** There is a route but no position has been supplied. */
    CtNavigationStateNoPosition,
    /** There is a turn ahead. */
    CtNavigationStateTurn,
    /** The current position is off the route. */
    CtNavigationStateOffRoute,
    /** A route is being calculated. */
    CtNavigationStateRouting,
    /** A new route has been calculated. */
    CtNavigationStateNewRoute,
    /** The latest position is on the route but a U-turn is needed. */
    CtNavigationStateTurnRound,
    /** The latest position is on the route and very close to the destination. */
    CtNavigationStateArrival,
    /** Navigation has started on a new route. */
    CtNavigationStateDeparture
    };
/** A synonym of CtNavigationState. */
typedef CtNavigationState CartoTypeNavigationState;

typedef NS_OPTIONS(uint32_t,CtFollowMode)
    {
    /** A flag to make the center of the map follow the user's location. */
    CtFollowModeLocationFlag = 1,
    /** A flag to rotate the map to the user's heading. */
    CtFollowModeHeadingFlag = 2,
    /** A flag to set the map to a suitable zoom level for the user's speed. */
    CtFollowModeZoomFlag = 4,
    /** The map does not follow the user's location or heading. */
    CtFollowModeNone = 0,
    /** The map is centred on the user's location. */
    CtFollowModeLocation = CtFollowModeLocationFlag,
    /** The map is centred on the user's location and rotated to the user's heading. */
    CtFollowModeLocationHeading = CtFollowModeLocationFlag | CtFollowModeHeadingFlag,
    /** The map is centred on the user's location and zoomed to a suitable level for the user's speed. */
    CtFollowModeLocationZoom = CtFollowModeLocationFlag | CtFollowModeZoomFlag,
    /** The map is centred on the user's location, rotated to the user's heading, and zoomed to a suitable level for the user's speed. */
    CtFollowModeLocationHeadingZoom = CtFollowModeLocationFlag | CtFollowModeHeadingFlag | CtFollowModeZoomFlag
    };
/** A synonym of CtFollowMode. */
typedef CtFollowMode CartoTypeFollowMode;

/** Information about the nearest route segment to a point. */
typedef struct
    {
    /** The index of the segment in the CtRoute object, or -1 if there were no segments. */
    int segmentIndex;
    /**
    The index of the line within the segment's path:
    line N goes from point N to point N + 1.
    */
    int lineIndex;
    /** The point in the segment's path nearest to the other point, in map coordinates. */
    CtPoint nearestPoint;
    /** The distance from the other point to iNearestPoint in meters. */
    double distanceToRoute;
    /** The distance of the nearest point along the route in meters. */
    double distanceAlongRoute;
    /** The distance within the current segment in meters. */
    double distanceAlongSegment;
    /** The estimated time of the nearest point, along the route, in seconds. */
    double timeAlongRoute;
    /** The estimated time within the current segment, in seconds. */
    double timeAlongSegment;
    /** The heading of the nearest line as a map angle taken anti-clockwise from rightwards. */
    double heading;
    } CtNearestSegmentinfo;
/** A synonym of CtNearestSegmentinfo. */
typedef CtNearestSegmentinfo CartoTypeNearestSegmentinfo;

/** A route segment. All segments except the first start at a turn (a junction). */
@interface CtRouteSegment: NSObject <CtPath>
    /** Initializes a CtRouteSegment with a C++ route segment object; for internal use only. */
    -(id)initWithRouteSegmentObject:(void*)aRouteSegmentObject;
    /** The road type.  */
    -(CtFeatureInfo*)featureInfo;
    /** The road name. */
    -(NSString*)name;
    /** The road reference. */
    -(NSString*)ref;
    /** The destination of the road. */
    -(NSString*)destination;
    /** Returns the distance of the segment in meters. */
    -(double)distance;
    /** Returns the estimated time taken to traverse the segment in seconds, including the turn time returned by the turnTime method. */
    -(double)time;
    /** Returns the estimated time in seconds for the turn at the start of the segment; this is included in the value returned by the time method. */
    -(double)turnTime;
    /** Returns the turn type. */
    -(CtTurnType)turnType;
    /** Returns true if this is a 'continue' turn, false if not. */
    -(bool)isContinue;
    /** Returns the roundabout state. */
    -(CtRoundaboutState)roundaboutState;
    /** Returns the turn angle. */
    -(double)turnAngle;
    /** Returns the exit number, counting the current junction as 0, if the junction at the start of the segment is part of a roundabout. */
    -(int)exitNumber;
    /** Returns the number of choices at this turning if known; 0 if not known. */
    -(int)choices;
    /** Returns the number of contours (sub-paths) in the path. */
    -(int)contourCount;
    /** Returns the number of points in a certain contour. */
    -(int)pointCount:(int)aContourIndex;
    /** Returns a point, given its contour index and point index. */
    -(CtPathPoint)point:(int)aContourIndex pointIndex:(int)aPointIndex;
    /** Returns the route section this route segment belongs to. */
    -(int)section;
    /** Returns turn instructions provided by the routing system. If not empty they are used instead of instructions generated from the information in this CtRouteSegment object. */
    -(NSString*)instructions;
    @end
/** A synonym of CtRouteSegment. */
@compatibility_alias CartoTypeRouteSegment CtRouteSegment;

/** A route. */
@interface CtRoute: NSObject <CtPath>
    /** Initializes a CtRoute with a C++ route object; for internal use only. */
    -(id)initWithRouteObject:(void*)aRouteObject;
    /** Returns the geometry of the route. */
    -(CtGeometry*)geometry;
    /** Returns the distance of the route in meters. */
    -(double)distance;
    /** Returns the total distance of any parts of the route on toll roads, in meters. */
    -(double)tollRoadDistance;
    /** Returns the estimated time taken to traverse the route in seconds. */
    -(double)time;
    /** Returns the number of route segments. */
    -(int)routeSegmentCount;
    /** Returns a route segment by its index. */
    -(CtRouteSegment*)routeSegmentByIndex:(int)aIndex;
    /**
    Gets information about the nearest route segment to a point given in map coordinates.
     
    The parameter aSection gives the current route section (a part of the route between waypoints;
    simple routes have a single section). The returned segment will always be in the current
    section or a following section. It is theoretically possible for a section to be completely skipped if it is very short or
    of zero length, so there is no constraint that section 0 must be followed by section 1 and not by section 2, etc.
    If aSection is negative it is ignored and the returned segment may be in any section.
     
    The parameter aPreviousDistanceAlongRoute should be provided (set to a value greater than zero)
    for multi-section routes, so that the correct route segment can be chosen if the route doubles back on itself.
    Positions further along the route will be given preference over those earlier in the route.
    */
    -(void)nearestSegment:(CtPoint)aPoint info:(CtNearestSegmentinfo*)aInfo section:(int)aSection prevDistance:(double)aPreviousDistanceAlongRoute;
    /** Gets information about a point a certain distance along a route. */
    -(void)pointAtDistance:(double)aDistanceInMeters info:(CtNearestSegmentinfo*)aInfo;
    /** Gets information about a point a certain estimated time along a route. */
    -(void)pointAtTime:(double)aTimeInSeconds info:(CtNearestSegmentinfo*)aInfo;
    /** Returns the number of contours (sub-paths) in the path. */
    -(int)contourCount;
    /** Returns the number of points in a certain contour. */
    -(int)pointCount:(int)aContourIndex;
    /** Returns a point, given its contour index and point index. */
    -(CtPathPoint)point:(int)aContourIndex pointIndex:(int)aPointIndex;
    /** Returns the internal C++ route object. */
    -(void*)route;
@end
/** A synonym of CtRoute. */
@compatibility_alias CartoTypeRoute CtRoute;

/**
Traffic information.
This information is normally used in combination with a location reference.
*/
typedef struct
    {
    /** Vehicle types affected by this information, taken from the KRouteAccess... bit definitions; use 0, or alternatively KRouteAccessVehicle, to select all vehicle types. */
    uint32_t vehicleTypes;
    /** Permitted or expected speed in kph: 255 or greater means no speed limit; 0 or less means the route is forbidden. */
    double speed;
    } CtTrafficInfo;
/** A synonym of CtTrafficInfo. */
typedef CtTrafficInfo CartoTypeTrafficInfo;

/** The orientation of a path along a road: used in traffic information. */
typedef NS_ENUM(NSInteger,CtRoadOrientation)
    {
    CtRoadOrientationNone,
    CtRoadOrientationForwards,
    CtRoadOrientationBackwards,
    CtRoadOrientationBoth,
    };
/** A synonym of CtRoadOrientation. */
typedef CtRoadOrientation CartoTypeRoadOrientation;

/** The type of a location reference used for traffic information. */
typedef NS_ENUM(NSInteger,CtLocationRefType)
    {
    /** A line in the route network. */
    CtLocationRefTypeLine,
    /** A point on the earth's surface. */
    CtLocationRefTypeGeoCoordinate,
    /** A point on a line in the route network. */
    CtLocationRefTypePointAlongLine,
    /** A point on the route network providing access to a nearby POI: the first point is the point on the line, the second is the POI. */
    CtLocationRefTypePointWithAccessPoint,
    /** A circle defined as a point and a radius. */
    CtLocationRefTypeCircle,
    /** A rectangle aligned to the grid of latitude and longitude: it is defined using two points at opposite corners. */
    CtLocationRefTypeRectangle,
    /** A polygon defined using a set of points. */
    CtLocationRefTypePolygon,
    /** A closed line in the route network. */
    CtLocationRefTypeClosedLine
    };
/** A synonym of CtLocationRefType. */
typedef CtLocationRefType CartoTypeLocationRefType;

/** Parameters used for a location reference when writing traffic information. */
typedef struct
    {
    /** The type of this location reference. */
    CtLocationRefType type;
    /** The arbitrary ID of the location reference. */
    __unsafe_unretained NSString* identifier;
    /** The point or points. */
    __unsafe_unretained CtGeometry* geometry;
    /** The radius, if this is a circle. */
    double radiusInMeters;
    /** The side of the road, if relevant. */
    CtSideOfRoad sideOfRoad;
    /** The road orientation, if relevant. */
    CtRoadOrientation roadOrientation;
    } CtLocationRef;
/** A synonym of CtLocationRef. */
typedef CtLocationRef CartoTypeLocationRef;

/** Parameters governing navigation behavior. */
@interface CtNavigatorParam: NSObject
    /** Minimum distance between location fixes in metres that is taken as an actual move. */
    @property (nonatomic) int minimumFixDistance;
    /** Maximum distance from the route in metres before the vehicle is deemed off-route. */
    @property (nonatomic) int routeDistanceTolerance;
    /** Maximum time off route in seconds before a new route needs to calculated. */
    @property (nonatomic) int routeTimeTolerance;
    /**
    If true, and if there is a route, the position on route is updated and turn information is created, when a position update arrives.
    If false, the vehicle position and speed are updated but other behaviour is the same as if there is no route.
    */
    @property (nonatomic) bool navigationEnabled;
    /** (for internal use) Returns a pointer to the underlying C++ object. */
    -(void*)navigatorParam;
@end
/** A synonym of CtNavigatorParam. */
@compatibility_alias CartoTypeNavigatorParam CtNavigatorParam;

/** A matrix of route times and distances between sets of points. */
@interface CtTimeAndDistanceMatrix: NSObject
    /** Returns the number of 'from' points. */
    @property (nonatomic, readonly) int fromCount;
    /** Returns the number of 'to' points. */
    @property (nonatomic, readonly) int toCount;
    /** Returns the time in seconds taken to travel on the best route from one of the 'from' points to one of the 'to' points. */
    -(int)time:(int)aFromIndex to:(int)aToIndex;
    /** Returns the distance in metres along the best route from one of the 'from' points to one of the 'to' points. */
    -(int)distance:(int)aFromIndex to:(int)aToIndex;
    /** (for internal use) Initialises the object from a C++ object. */
    -(id)initWithMatrix:(void*)aMatrix;
@end
/** A synonym of CtTimeAndDistanceMatrix. */
@compatibility_alias CartoTypeTimeAndDistanceMatrix CtTimeAndDistanceMatrix;

#endif
